///////////////////////////////////////////////////////////////////////////////
//	common_simod.cpp	: this is for SI daughter modules
//
//	Description:
//		Contains simple common routines to all daughter cards,
//
//	Revision History:
//		2003-01-28: Colin
//			Created  
//		2003-05-28: Mik
//			Changed calls for PLXC6711 to work with new API.
//		2003-06-19: Mik
//			Using extra params for COFF load/verify.
//		2004-08-16: Ley
//			Separated from old mod64.cpp, it now 
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "../../siddk_plx/apps/sihw_plx/Rev1/common/sisample.h"

#ifdef MOD64
#include "../mod64/mod64.h"
#elif MOD68
#include "../mod68/mod68.h"
#endif
//#include "common_simod.h"
#include "../../common_sihw/cboard.h"

//global variables
UINT32 isCoffLoaded = FALSE;

//extern CPLXBoard *board;
extern struct PLXC33 board;

///////////////////////////////////////////////////////////////////////////////
//	Sample_ReadADCsRAW
//
//	Description: Read ADC Data (16 bit 2's complement)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//		UINT32 data[64]:
//			Output array containing 64 values corresponding to all
//			64 ADC channels.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-19 Ley.
//			Added comments.
//
///////////////////////////////////////////////////////////////////////////////
//reads the AD registers as 2's complement 16bit signed integers
INT32 read_AD
(
	struct PLXDevice *pPLXDev, int commMode, UINT32 data[64]
)
{
	INT32 error;

	error = SI_Read
	(
		pPLXDev, commMode, 64, 
		kDaughterAddr, data
	);

	if (error != e_Err_NoError)
	{
		printf ("\nREAD FAILED!\n");
			return e_Err_ReadError;
	}	
	
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	write_DAC
//
//	Description: Write to DAC register selected by dac_num 0 to 3
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//		UINT32 dac_num:
//			Number of DAC to be written.
//		UINT32 *data:
//			Pointer to data to be written to DAC.
//		UINT32 daughterAddr:
//			UINT32 daughterOffsetDAC, UINT32 maxDACs
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-19 Ley.
//			Added comments.
//
///////////////////////////////////////////////////////////////////////////////
INT32 write_DAC
(
	struct PLXDevice *pPLXDev, int commMode, UINT32 dac_num, UINT32 *data,
	UINT32 daughterAddr, UINT32 daughterOffsetDAC, UINT32 maxDACs
)
{
	INT32 error;

	if(dac_num > (maxDACs - 1))
		return e_Err_CountTooBig;

	error = SI_Write
	(
		pPLXDev, commMode, 1, 
		daughterAddr + ((daughterOffsetDAC  + dac_num) * kDSPBF), data
	);

	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}	
	
	return e_Err_NoError;
}

//////////////////////////////////////////////////////////////////////////
//write to DIO register (8bit)
INT32 write_DIO
(
	struct PLXDevice *pPLXDev, int commMode, UINT32 *data,
	UINT32 daughterAddr, UINT32 daughterOffsetDIO
)
{
	INT32 error;

	error = SI_Write
	(
		pPLXDev, commMode, 1, 
		daughterAddr + (daughterOffsetDIO * kDSPBF), data
	);

	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}	
	
	return e_Err_NoError;
}


//////////////////////////////////////////////////////////////////////////
//gerneric (C6711 and C33) write to SI board
INT32 SI_Write
(
	struct PLXDevice *pPLXDev, int commMode,
	UINT32 count, UINT32 dsp_addr, UINT32 *host_addr
)
{
	INT32 error;

	if(!commMode)
	{
		error = SI_PLXC33_WriteTarget
		(
			pPLXDev,  0, count, 
			dsp_addr, host_addr
		);
	}
	else
	{
		error = SI_PLXC33_WriteHostpoll
		(
			pPLXDev, count, 
			dsp_addr, host_addr
		);
	}	

	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}	

	return e_Err_NoError;
}

//////////////////////////////////////////////////////////////////////////
//gerneric (C6711 and C33) read from SI board
INT32 SI_Read
(
	struct PLXDevice *pPLXDev, int commMode,
	UINT32 count, UINT32 dsp_addr, UINT32 *host_addr
)
{
	INT32 error;

#ifdef SIDEF_PLX_C3x
	if(!commMode)
	{
		error = SI_PLXC33_ReadTarget
		(
			pPLXDev,  0, count, 
			dsp_addr, host_addr
		);
	}
	else
	{
		error = SI_PLXC33_ReadHostpoll
		(
			pPLXDev, count, 
			dsp_addr, host_addr
		);
	}
#endif
	
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}	

	return e_Err_NoError;
}

//////////////////////////////////////////////////////////////////////////
//
//	Function used to calculate DDS phase accumulator values
//

INT32 SampleFrequencyComputation
(
	UINT32 ddsOutSampleRate,
	float refClockIn,
	UINT32 sampleRate,
	UINT32 paResolution,
	UINT32 *phaseAccumulatorArray,
	UINT32 *phseAccumulator,
	UINT32 *counterValue16bits
)
{
	float fTemp;

	fTemp = 1 / refClockIn;
	fTemp = fTemp * ((float) pow(2, paResolution));

	fTemp = (float) ddsOutSampleRate * sampleRate * fTemp;
	if(fmod(fTemp, 1) >= .5) fTemp += 1;
	*phseAccumulator = (UINT32) fTemp;

	phaseAccumulatorArray[0] = (*phseAccumulator >> 24) & 0xFF;
	phaseAccumulatorArray[1] = (*phseAccumulator >> 16) & 0xFF;
	phaseAccumulatorArray[2] = (*phseAccumulator >> 8) & 0xFF;
	phaseAccumulatorArray[3] = *phseAccumulator & 0xFF;

	fTemp = refClockIn / (sampleRate * 2);
	*counterValue16bits = (((UINT32)fTemp) - 1) & 0xFFFF;

	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_WriteCalFiletoDSP
//
//	Description: Writes contents in Calibration File into DSP, so calibration
//		can be done by DSP at real time. This way of applying calibration is
//		a special way for specific customers only, not present in the normal
//		perfomance of the DSP.
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_WriteCalFiletoDSP(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	char calFilePath[256], tempChar[kMaxCalFileSize], *tempChar2, *tempChar3;
	FILE *fHandle;
	UINT32 count, cnt;
	float calDataFloat[1024];		// Needed to use Conv_Float_Host2DSP.
	UINT32 calDataUINT32[1024];		// Needed to write Cal Data into DSP (WriteDSPActive);
	INT32 error;
	UINT32 caladdr = CalAddr;

	printf("Enter the full path for the Cal FILE (example = cal.tbl):\n");
	scanf("%s", calFilePath);
	getchar();

	fHandle = fopen(calFilePath, "rt");
	if(!fHandle)
	{
		printf("Error Opening Calibration File\n");
		return;
	}

	count = fread(tempChar, sizeof(char), kMaxCalFileSize, fHandle);
	if(!count)
	{
		printf("Error reading Calibration File\n");
		return;
	}
	if(count == kMaxCalFileSize)
	{
		printf("Error, Calibration File too big\n");
		return;
	}

	// extract Cal Data to Float format.
	tempChar2 = tempChar;
	for(cnt = 0; cnt < 1024; cnt++)
	{
		// convert ascii characters to floating point format.
		calDataFloat[cnt] = (float) strtod(tempChar2, &tempChar3);
		// if tempChar2 == tempChar3, means there's no more
		// valid numbers in file.
		if(tempChar2 == tempChar3)
		{
			break;
		}
		tempChar2 = tempChar3;
	}
	count = cnt;

	// Apply Scale Factor for Gains.
	for(cnt = 0; cnt < 128; cnt++)
	{
		if(count > 128 + cnt)
			calDataFloat[128 + cnt] *= kScaleFactor;
		if(count > 384 + cnt)
			calDataFloat[384 + cnt] /= kScaleFactor;
		if(count > 640 + cnt)
			calDataFloat[640 + cnt] *= kScaleFactor;
		if(count > 896 + cnt)
			calDataFloat[896 + cnt] /= kScaleFactor;
	}

	Conv_Float_Host2DSP(count, calDataFloat, calDataUINT32);

	error = SI_Write
		(
			pPLXDev,  
			1, count, caladdr, calDataUINT32
		);
	if(error)
	{
		printf("Error Writting Calibration File to DSP\n");
		return;
	}

	printf("Writting Calibration File to DSP Success\n");                    
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_ReadADCsRAW
//
//	Description: Read ADC Data (16 bit 2's complement)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_ReadADCsRAW(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	UINT32 data[64], cnt;

	if(read_AD(pPLXDev, commMode, data)  != e_Err_NoError)
	{
		printf("Read Error!\n");
	}
	else
	{
		for(cnt = 0; cnt < 64; cnt++)
			printf("ADC %d: 0x%x\n", cnt, (UINT16)data[cnt]);
	}
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_WriteDACsRAW
//
//	Description: Write DAC Data (16 bit 2's complement)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_WriteDACsRAW(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	UINT32 data[64];
	int dac_num;

	printf("Select DAC Channel [0-15]: ");
	scanf("%d", &dac_num);
	printf("Enter data (2's complement): 0x");
	scanf("%x", &data[0]);
	
	if
	(
		write_DAC(pPLXDev, commMode, dac_num, data,
			kDaughterAddr, kDaughterOffsetDAC, kDaughterNumDAC)
		!= e_Err_NoError
	)
		printf("Write Error!\n");
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_ReadADCswCal
//
//	Description: Read ADCs with DSP Calibration included (Voltage format)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////

void Sample_ReadADCswCal(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	UINT32 dspData[64], cnt;
	float ieeeData[64];

	if(read_ADwCal(pPLXDev, commMode, dspData)  != e_Err_NoError)
	{
		printf("Read Error!\n");
	}
	else
	{
		Conv_Float_DSP2Host(64, dspData, ieeeData);
		for(cnt = 0; cnt < 64; cnt++)
			printf("ADC %d: %f V\n", cnt, ieeeData[cnt]);
	}
}


///////////////////////////////////////////////////////////////////////////////
//	read_ADwCal
//
//	Description: Read ADCs with DSP Calibration included (Voltage format)
//		It requests DSP to read 0x400 locations displaced from normal ADC
//		locations, by doing so, DSP responds reading the normal addresses,
//		then apply calibration before passing data to Host.
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
//reads the AD registers as calibrated voltage
INT32 read_ADwCal
(
	struct PLXDevice *pPLXDev, int commMode, UINT32 data[64]
)
{
	INT32 error;

	error = SI_Read
	(
		pPLXDev, commMode, 64, 
		kDaughterAddr + 0x400 * kDSPBF, data
	);

	if (error != e_Err_NoError)
	{
		printf ("\nREAD FAILED!\n");
			return e_Err_ReadError;
	}	
	
	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_WriteDACswCal
//
//	Description: Write DACs with DSP Calibration included (Voltage format)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_WriteDACswCal(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	float data;
	int dac_num;

	printf("Select DAC Channel [0-15]: ");
	scanf("%d", &dac_num);
	printf("Enter data (Voltage, floating point format): ");
	scanf("%f", &data);
	
	if
	(
		write_DACwCal(pPLXDev, commMode, dac_num, &data,
			kDaughterAddr, kDaughterOffsetDAC, kDaughterNumDAC)
		!= e_Err_NoError
	)
		printf("Write Error!\n");
}


///////////////////////////////////////////////////////////////////////////////
//	read_DACwCal
//
//	Description: Write DACs with DSP Calibration included (Voltage format)
//		It requests DSP to write 0x400 locations displaced from normal DAC
//		locations, by doing so, DSP responds by applying calibration
//		before it actually write to the normal DAC addresses,
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
//write to DAC register selected by dac_num 0 to 3
INT32 write_DACwCal
(
	struct PLXDevice *pPLXDev, int commMode, UINT32 dac_num, float *data,
	UINT32 daughterAddr, UINT32 daughterOffsetDAC, UINT32 maxDACs
)
{
	INT32 error;
	UINT32 tempUINT32;

	if(dac_num > (maxDACs - 1))
		return e_Err_CountTooBig;

	Conv_Float_Host2DSP(1, data, &tempUINT32);

	error = SI_Write
	(
		pPLXDev, commMode, 1, 
		daughterAddr + ((daughterOffsetDAC  + dac_num + 0x400) * kDSPBF), &tempUINT32
	);

	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}	
	
	return e_Err_NoError;
}


