///////////////////////////////////////////////////////////////////////////////
//	mod68.cpp	: this is the filename for SI daughter module 6800 program .
//
//	Description:
//		Contains the basic functions to calculate the configuration params
//		for the MOD 6800 board. Also has simple demo routines to read and
//		write to the daughter module registers.
//
//	Revision History:
//		2003-01-28: Colin
//			Created  
//		2003-05-28: Mik
//			Changed calls for PLXC6711 to work with new API.
//		2003-06-19: Mik
//			Using extra params for COFF load/verify.
//		2004-08-16: Ley
//			Copied and modified from mod64.cpp, but also removed common
//			routines for daughter cards.
//		2006-10-16: Ley
//			Expanded GetBoardInfo so now it also outputs
//			DSP Bus Speed and Boundary Factor.
//			For SISample related programs: Added read NVRam to have the DSP
//			Bus speed in order to calculate parameters for DDS
//			for Sampling Rate.
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>

#ifdef REV0
#include "../../siddk_plx/apps/sihw_plx/Rev0/common/sisample.h"
#else
#include "../../siddk_plx/apps/sihw_plx/Rev1/common/sisample.h"
#endif

#include "mod68.h"
#include "../../common_sihw/cboard.h"
#include "../../common_ddk/os_lnx.h"

extern UINT32 isCoffLoaded;
//extern INT32 OS_GetTickCount();

///////////////////////////////////////////////////////////////////////////////
//	Sample_6800
//
//	Description: Function to call from the main command loop
//				 in SISAMPLE program
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 dspBusClkSpeed:
//			DSP Bus Clock Speed.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-019 Ley.
//			Added comments.
//
//////////////////////////////////////////////////////////////////////////
void Sample_6800(struct PLXDevice *pPLXDev, UINT32 dspBusClkSpeed)
{
	bool Done = false;
	char user_input[256];
	UINT32 commMode;
	T_6800ConfigParams mod, *pmod = &mod;

	//You can use any available communications modes. 
	//currently the code only supports hostpoll and target
	//default to hostpoll for c6711
	commMode = 1;

	//forces the user to coff load
	if(!isCoffLoaded)
	{
		Sample_LoadCofffile();

		isCoffLoaded = TRUE;
	}
//end of forcing the user to coff load

	while(!Done)
	{
		printf(kDMENU);
		scanf("%s", user_input);
		getchar();
		
		switch(user_input[0])
		{
		case '1':	//Set Default Configuration (Clocks, Muxes, etc.)
			Sample_6800_Config
			(
				pPLXDev, 
				pmod,
				commMode,
				dspBusClkSpeed
			);
			break;
		case '2':	//Transfer Cal PROM contents to Cal Table (SI-MOD66 only)
			Sample_6800_CalConfig(pPLXDev, commMode);

			break;
		case '3':	//Read ADC Data (16 bit 2's complement)
			Sample_6800_ReadADCs(pPLXDev, commMode);

			break;
		case '4':	//Write to DAC
			Sample_6800_WriteDACs(pPLXDev, commMode);

			break;
		case '5':	//Digital I/O
			Sample_6800_RWDititalIO
			(
				pPLXDev,
				pmod,
				commMode
			);

			break;
		case '0':
			printf("\nExiting 6800 Menu......\n");
			Done = true;
			break;

		default:
			printf("\nInvalid Option. Press Enter To Continue.");
			getchar();
			break;		
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
//	default_configure_MOD68
//
//	Description: Sets a default config for the daughter module.
//		This can be changed to reflect a user prefered setting.
//		Routines for making it dynamic can also be written, the
//		function set68_clksrc is an example
//
//	Inputs:
//		T_6800ConfigParams *mod:
//			Pointer to structure of parameters for MOD6800.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-019 Ley.
//			Added comments.
//
//////////////////////////////////////////////////////////////////////////
void default_configure_MOD68(T_6800ConfigParams *mod)
{
	UINT32 i, mx, cnt;

// assign ADC Parameters
	mod->adcParams.channelsMux = 16;

	for(mx = 0; mx < kDaughterNumMux; mx++)
	{
		for(i = 0; i < kDaughterMaxChannels/2; i++)
		{
			mod->adcParams.mux[mx][i].dg = dg_one;
			mod->adcParams.mux[mx][i].sg = sg_one;
			mod->adcParams.mux[mx][i].input = singleA;
		}
		for(i = kDaughterMaxChannels/2; i < kDaughterMaxChannels; i++)
		{
			mod->adcParams.mux[mx][i].dg = dg_one;
			mod->adcParams.mux[mx][i].sg = sg_one;
			mod->adcParams.mux[mx][i].input = singleB;
		}
	}

// assign Timing Parameters
	mod->timingParams.sampleRateValues.sampleRateDDS0 = 5000;
	mod->timingParams.sampleRateValues.sampleRateDDS1 = 5000;
	mod->timingParams.sampleRateValues.sampleRateExtDDS = 5000;
	mod->timingParams.sampleRateValues.maxADCRate = 250000;

	mod->timingParams.sampleRateManagement.extClockEnable = 0;
	mod->timingParams.sampleRateManagement.sampleMode = 0;
	mod->timingParams.sampleRateManagement.dds0TriggerMode = 0;
	mod->timingParams.sampleRateManagement.dds1TriggerMode = 0;
	mod->timingParams.sampleRateManagement.sampleClockSource = 0;

	for(cnt = 0; cnt < 4; cnt ++)
	{
		mod->timingParams.ecParams[cnt].ecReset = 0;
		mod->timingParams.ecParams[cnt].ecEnable = 0;
		mod->timingParams.ecParams[cnt].ecWrapMode = 0;
		mod->timingParams.ecParams[cnt].ecPolarity = 0;
		mod->timingParams.ecParams[cnt].ecTickClockSource = 0;
		mod->timingParams.ecParams[cnt].ecThreshold = 1;
	}

// assign GPIO Parameters
	mod->gpIOParams.gpIOConfig.gpioFunction1_0 = 0;
	mod->gpIOParams.gpIOConfig.gpioFunction3_2 = 0;
	mod->gpIOParams.gpIOConfig.freeCounterSource = 0;

	mod->gpIOParams.auxDIO = 0;

	mod->gpIOParams.pulseParams.pulseIn0Reset = 0;
	mod->gpIOParams.pulseParams.pulseIn1Reset = 0;
	mod->gpIOParams.pulseParams.pulseOut0Reset = 0;
	mod->gpIOParams.pulseParams.pulseOut1Reset = 0;
	mod->gpIOParams.pulseParams.pulseOut0HighTickCount = 0;
	mod->gpIOParams.pulseParams.pulseOut0LowTickCount = 0;
	mod->gpIOParams.pulseParams.pulseOut1HighTickCount = 0;
	mod->gpIOParams.pulseParams.pulseOut1LowTickCount = 0;

	mod->gpIOParams.qeParams.QE0DivideRatio = 0;
	mod->gpIOParams.qeParams.QE0CountMode = 0;
	mod->gpIOParams.qeParams.QE0IndexMode = 0;
	mod->gpIOParams.qeParams.QE1DivideRatio = 0;
	mod->gpIOParams.qeParams.QE1CountMode = 0;
	mod->gpIOParams.qeParams.QE1IndexMode = 0;

// assign DAC Clock and Digital IO Port Direction parameters
	mod->dacClock = 0;
	mod->digitalIOPortDirection = 0;
}

///////////////////////////////////////////////////////////////////////////////
//	setup68_DaughterMod
//
//	Description: Write all the parameters from the T_6800ConfigParams struct
//		to the daughter module commMode is the desired communications method
//		0 = Target; 1 = Hostpoll; after the mod struct has been setup with
//		the desired configuration this function must be used to call the
//		calculation functions and do some calculations of its own and write
//		the results to the appropriate config registers for the MOD 6800 board.
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		T_6800ConfigParams *mod:
//			Pointer to structure of parameters for MOD6800.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//		UINT32 dspBusClkSpeed:
//			DSP Bus Clock Speed.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-019 Ley.
//			Added comments.
//
//////////////////////////////////////////////////////////////////////////
INT32 setup68_DaughterMod
(
	struct PLXDevice *pPLXDev,
	T_6800ConfigParams *mod,
	UINT32 commMode,
	UINT32 dspBusClkSpeed
)
{
	UINT32 adcSampleRates[4], channelParams[32], dacCSR, timingCSRs[9],
		gpioCSR, auxDIOCSR, qeCSR, pulseCSR[13], dioCSR;

	INT32 error;
	UINT32 phaseAccumulatorArray[4], phaseAccumulator, counterValue16bits;
	UINT32 cnt, temp, tempArray[5];

	if(commMode > 1)
		return e_Err_WriteError;

	calc68_IOParams
	(
		mod,
		adcSampleRates,
		channelParams,
		&dacCSR,
		timingCSRs,
		&gpioCSR,
		&auxDIOCSR,
		&qeCSR,
		pulseCSR,
		&dioCSR
	);

	// 0: ADC Section: Load Param/Scan Table
	temp = kDaughterAddr + kDSP68xxScanTable * kDSPBF;
	error = SI_Write(pPLXDev, 1, 16, temp, channelParams);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 1: Timing Section
	// 1.0: INT_DDS0_PADATA: Write Control/Phase Info
	SampleFrequencyComputation
	(
		1,
		(float) (dspBusClkSpeed / 2),
		adcSampleRates[0],
		27,
		phaseAccumulatorArray,
		&phaseAccumulator,
		&counterValue16bits
	);

	temp = kDaughterAddr + kDSP68xxIntDDS0PA * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &phaseAccumulator);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 1.1: INT_DDS1_PADATA: Write Control/Phase Info
	SampleFrequencyComputation
	(
		1,
		(float) (dspBusClkSpeed / 2),
		adcSampleRates[1],
		27,
		phaseAccumulatorArray,
		&phaseAccumulator,
		&counterValue16bits
	);

	temp = kDaughterAddr + kDSP68xxIntDDS1PA * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &phaseAccumulator);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 1.2: Ext_DDS
	// 1.2.0: Timing CSR: Clear Ext_DDS
	temp = kDaughterAddr + kDSP68xxTimerCSR * kDSPBF;

	cnt = 1;
	error = SI_Write(pPLXDev, 1, 1, temp, &cnt);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}
	cnt = 0;
	error = SI_Write(pPLXDev, 1, 1, temp, &cnt);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 1.2.1: Ext_DDS_PADATA: Write Control/Phase Info
	SampleFrequencyComputation
	(
		2,
		(float) (dspBusClkSpeed / 2),
		adcSampleRates[2],
		32,
		phaseAccumulatorArray,
		&phaseAccumulator,
		&counterValue16bits
	);

	tempArray[0] = 0;
	for(cnt = 0; cnt < 4; cnt++)
	{
		tempArray[1 + cnt] = phaseAccumulatorArray[cnt];
	}

	temp = kDaughterAddr + kDSP68xxExtDDSPA * kDSPBF;
	for(cnt = 0; cnt < 5; cnt++)
	{
		error = SI_Write(pPLXDev, 1, 1, temp, &tempArray[cnt]);
		if (error != e_Err_NoError)
		{
			printf ("\nWRITE FAILED!\n");
				return e_Err_WriteError;
		}
	}

	// 1.2.2: Ext_DDS_LOAD: Load Phase
	cnt = 0;
	temp = kDaughterAddr + kDSP68xxExtDDSLD * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &cnt);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 1.3: EC[3:0] Thresholds
	temp = kDaughterAddr + kDSP68xxRegOffset * kDSPBF;
	error = SI_Write(pPLXDev, 1, 4, temp, &timingCSRs[5]);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 1.4: Timing/EC CSRs
	// 1.4.0: EC_CSRs: Clear EC
	temp = kDaughterAddr + kDSP68xxClearECCSR * kDSPBF;
	for(cnt = 0; cnt < 4; cnt++)
	{
		tempArray[cnt] = 1;
	}
	error = SI_Write(pPLXDev, 1, 4, temp, &tempArray[0]);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 1.4.1: Load CSRs
	temp = kDaughterAddr + kDSP68xxTimerCSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 5, temp, &timingCSRs[0]);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 2: DAC Section
	temp = kDaughterAddr + kDSP68xxDACCSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &dacCSR);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 3: DIO Section
	temp = kDaughterAddr + kDSP68xxDIOCSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &dioCSR);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 4: GPIO Section
	// 4.0: Aux DIO CSR
	temp = kDaughterAddr + kDSP68xxAuxDIOCSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &auxDIOCSR);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 4.1.0: Clear Pulses
	cnt = 0xF;
	temp = kDaughterAddr + kDSP68xxPulseCSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &cnt);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 4.1.1: Update Pulse IOs CSR
	temp = kDaughterAddr + (kDSP68xxPulseCSR + 5) * kDSPBF;
	error = SI_Write(pPLXDev, 1, 13 - 5, temp, &pulseCSR[5]);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 4.1.2: Update Pulse CSR
	temp = kDaughterAddr + kDSP68xxPulseCSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &pulseCSR[0]);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 4.2: QE CSR
	temp = kDaughterAddr + kDSP68xxQECSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &qeCSR);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	// 4.3: GPIO CSR
	temp = kDaughterAddr + kDSP68xxGPIOCSR * kDSPBF;
	error = SI_Write(pPLXDev, 1, 1, temp, &gpioCSR);
	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED!\n");
			return e_Err_WriteError;
	}

	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	calc68_IOParams
//
//	Description: Calculates all the parameters based on 
//		the mod struct passed in.
//
//	Inputs:
//		T_6800ConfigParams *mod:
//			Pointer to structure of parameters for MOD6800.
//		UINT32 *adcSampleRates:
//			Pointer to an array containing all sample rates.
//		UINT32 *channelParams:
//			Pointer to an array containing all channel parameters.
//		UINT32 *dacCSR:
//			Pointer to a value to be written to the DAC CSR.
//		UINT32 *timingCSRs:
//			Pointer to an array containing the values to be written
//			to the Timing CSRs.
//		UINT32 *gpioCSR:
//			Pointer to a value to be written to the GPIO CSR.
//		UINT32 *auxDIOCSR:
//			Pointer to a value to be written to the Aux. I/O CSR.
//		UINT32 *qeCSR:
//			Pointer to a value to be written to the Quadrature Encoder (QE)
//			counter CSR.
//		UINT32 *pulseCSR:
//			Pointer to a value to be written to the Pulse CSR.
//		UINT32 *dioCSR:
//			Pointer to a value to be written to the Digital I/O CSR.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-019 Ley.
//			Added comments.
//
//////////////////////////////////////////////////////////////////////////
//calculates all the Scan table parameters based on 
//the mod struct passed in.
void calc68_IOParams
(
	T_6800ConfigParams *mod,
	UINT32 *adcSampleRates,
	UINT32 *channelParams,
	UINT32 *dacCSR,
	UINT32 *timingCSRs,
	UINT32 *gpioCSR,
	UINT32 *auxDIOCSR,
	UINT32 *qeCSR,
	UINT32 *pulseCSR,
	UINT32 *dioCSR
)
{
	UINT32 cnt, evenLoc, oddLoc, chansMux, sampleRate[3];
	UINT32 temp, temp1, temp2;
	// 0: Calculate Channel Parameters
	for(cnt = 0; cnt < mod->adcParams.channelsMux; cnt++)
	{
		evenLoc = 0;
		evenLoc = ((((evenLoc) * 4 + mod->adcParams.mux[3][cnt].input) * 4
				+ mod->adcParams.mux[2][cnt].input) * 4
				+ mod->adcParams.mux[1][cnt].input) * 4
				+ mod->adcParams.mux[0][cnt].input;

		//mux A params
		evenLoc = (evenLoc * 256) + (cnt & 7);

		//gain parameters
		oddLoc = 0;
		oddLoc = ((((((((oddLoc) * 4 +
				mod->adcParams.mux[3][cnt].sg) * 4 +
				mod->adcParams.mux[2][cnt].sg) * 4 +
				mod->adcParams.mux[1][cnt].sg) * 4 +
				mod->adcParams.mux[0][cnt].sg) * 4 +
				mod->adcParams.mux[3][cnt].dg) * 4 +
				mod->adcParams.mux[2][cnt].dg) * 4 +
				mod->adcParams.mux[1][cnt].dg) * 4 +
				mod->adcParams.mux[0][cnt].dg;

		channelParams[cnt] = (oddLoc * 65536) + evenLoc;
	}

	//sync bit
	channelParams[mod->adcParams.channelsMux - 1] =
		channelParams[mod->adcParams.channelsMux - 1] + 8;

	// 1: DAC Section
    *dacCSR = mod->dacClock;

	// 2: Timing Section
	// 2.0: ADC Sample Clock Generation
	sampleRate[0] = mod->timingParams.sampleRateValues.sampleRateDDS0;
	sampleRate[1] = mod->timingParams.sampleRateValues.sampleRateDDS1;
	sampleRate[2] = mod->timingParams.sampleRateValues.sampleRateExtDDS;

	temp = mod->adcParams.channelsMux;
	if(mod->timingParams.sampleRateManagement.sampleMode == 1)
	{
		temp = 1;
	}

	for(cnt = 0; cnt < 3; cnt++)
	{
		chansMux = 1;
		if(mod->timingParams.sampleRateManagement.sampleClockSource == cnt)
		{
			chansMux = temp;
		}
		adcSampleRates[cnt] = sampleRate[cnt] * chansMux;
		if(adcSampleRates[cnt] > mod->timingParams.sampleRateValues.maxADCRate)
		{
			adcSampleRates[cnt] = mod->timingParams.sampleRateValues.maxADCRate;
		}
	}

	// 2.1: Timing CSR
	timingCSRs[0] =
		mod->timingParams.sampleRateManagement.extClockEnable * 2 +
		mod->timingParams.sampleRateManagement.sampleMode * 4 +
		mod->timingParams.sampleRateManagement.dds0TriggerMode * 8 +
		mod->timingParams.sampleRateManagement.dds1TriggerMode * 16 +
		mod->timingParams.sampleRateManagement.sampleClockSource * 512;

	// 2.2: EC CSRs
	for(cnt = 0; cnt < 4; cnt++)
	{
		timingCSRs[1 + cnt] =
			mod->timingParams.ecParams[cnt].ecReset +
			(mod->timingParams.ecParams[cnt].ecEnable << 1) +
			(mod->timingParams.ecParams[cnt].ecWrapMode << 2) +
			(mod->timingParams.ecParams[cnt].ecPolarity << 3) +
			(mod->timingParams.ecParams[cnt].ecTickClockSource << 8);

		timingCSRs[5 + cnt] =
			mod->timingParams.ecParams[cnt].ecThreshold;
	}

	// 3: Digital I/O Port Section
	*dioCSR = mod->digitalIOPortDirection;

	// 4: GP I/O Section
	// 4.0: Config GP I/O
	*gpioCSR =
		mod->gpIOParams.gpIOConfig.gpioFunction1_0 +
		mod->gpIOParams.gpIOConfig.gpioFunction3_2 * 4 +
		mod->gpIOParams.gpIOConfig.freeCounterSource * 16;

	// 4.1: Config Aux DIO
	*auxDIOCSR =
		mod->gpIOParams.auxDIO;

	// 4.2: Config Pulse
	pulseCSR[0] =
		mod->gpIOParams.pulseParams.pulseIn0Reset +
		mod->gpIOParams.pulseParams.pulseOut0Reset * 2 +
		mod->gpIOParams.pulseParams.pulseIn1Reset * 4 +
		mod->gpIOParams.pulseParams.pulseOut1Reset * 8;
	for(cnt = 0; cnt < 4; cnt++)
	{
		pulseCSR[1 + cnt] = 0;
		pulseCSR[7 + cnt] = 0;
	}
	pulseCSR[5] =
		mod->gpIOParams.pulseParams.pulseOut0HighTickCount;
	pulseCSR[6] =
		mod->gpIOParams.pulseParams.pulseOut0LowTickCount;
	pulseCSR[11] =
		mod->gpIOParams.pulseParams.pulseOut1HighTickCount;
	pulseCSR[12] =
		mod->gpIOParams.pulseParams.pulseOut1LowTickCount;

	// 4.3: Config QE
	temp1 = mod->gpIOParams.qeParams.QE0DivideRatio;
	if(temp1 > 15) temp1 = 15;
	temp2 = mod->gpIOParams.qeParams.QE1DivideRatio;
	if(temp2 > 15) temp2 = 15;
	*qeCSR =
		temp1 +
		(mod->gpIOParams.qeParams.QE0CountMode << 4) +
		(mod->gpIOParams.qeParams.QE0IndexMode << 7) +
		(temp2 << 16) +
		(mod->gpIOParams.qeParams.QE1IndexMode << 20) +
		(mod->gpIOParams.qeParams.QE1CountMode << 23);
}


///////////////////////////////////////////////////////////////////////////////
//	PollI2CBusySIMOD66
//
//	Description: Polls for flag in FPGA of SI-MOD6600 to detect when FPGA
//		is still busy transfering data from Cal PROM to Cal Table
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		INT32 timeout:
//			Timeout value.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-08-02 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////

INT32 PollI2CBusySIMOD66
(
	struct PLXDevice *pPLXDev,
	INT32 timeout,
	UINT32 commMode
)
{
	INT32 error;
	INT32 time1, time2;
	INT32 withinTimeout, busy;
	UINT32 flag;

	withinTimeout = 1;
	busy = 1;
	time1 = OS_GetTickCount();

	do
	{
		time2 = OS_GetTickCount();
		if((time2 - time1) > timeout)
		{
			withinTimeout = 0;
			break;
		}

		error = SI_Read
		(
			pPLXDev, commMode, 1,
			kDaughterAddr + kDSP68xxEEPROMtoFPGACal * kDSPBF, &flag
		);
		if(error) return error;
		if((flag & 3) == 0)
		{
			busy = 0;
		}
	}while(withinTimeout & busy);

	if(withinTimeout == 0)
	{
		return e_Err_Timeout;
	}

	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	Xfere2PROMtoFPGA
//
//	Description: Writes a bit into FPGA of SI-MOD6600 to trigger a transfer
//		from Cal PROM to FPGA Cal Table. Note that only Calibration data for
//		configured channels are transfered.
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-08-02 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////

INT32 Xfere2PROMtoFPGA
(
	struct PLXDevice *pPLXDev,
	UINT32 commMode
)
{
	INT32 timeout;
	INT32 error;
	UINT32 flag;

	timeout = 10000;			//10000 mS = 10S timeout.

	error = PollI2CBusySIMOD66(pPLXDev, timeout, commMode);
	if(error) return error;

	flag = 2;
	error = SI_Write
	(
		pPLXDev,  commMode, 1, 
		kDaughterAddr + (kDSP68xxEEPROMtoFPGACal * kDSPBF), &flag		
	);	

	error = PollI2CBusySIMOD66(pPLXDev, timeout, commMode);
	if(error) return error;

	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_6800_Config
//
//	Description: Set Default Configuration (Clocks, Muxes, etc.)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		T_6800ConfigParams pmod:
//			Pointer to 6800 Configuration Parameters.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//		UINT32 dspBusClkSpeed:
//			DSP Bus Clock Speed.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_6800_Config
(
	struct PLXDevice *pPLXDev, 
	T_6800ConfigParams *pmod,
	UINT32 commMode,
	UINT32 dspBusClkSpeed
)
{
	default_configure_MOD68(pmod);
	setup68_DaughterMod
	(
		pPLXDev, pmod, commMode, dspBusClkSpeed
	);
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_6800_CalConfig
//
//	Description: Transfer Cal PROM contents to Cal Table (SI-MOD66 only)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_6800_CalConfig(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	UINT32 selection;

	printf("0 = Transfer Cal FILE contents to DSP's Cal Table (All SI-MOD6x)\n");
	printf("1 = Transfer Cal PROM contents to FPGA's Cal Table (SI-MOD66 only)\n");

	scanf("%d", &selection);
	getchar();

	switch(selection)
	{
		case 0:

			Sample_WriteCalFiletoDSP(pPLXDev, commMode);

			break;

		case 1:

			Sample_6800_Xfere2PROMtoFPGA(pPLXDev, commMode);

			break;
		default:
			printf("Invalid Selection.\n");
			break;
	}
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_6800_ReadADCs
//
//	Description: Read ADC Data (16 bit 2's complement)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_6800_ReadADCs(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	UINT32 selection;

	printf("0 = Read raw ADC data (values returned as 16 bit 2's complement)\n");
	printf("1 = Read calibrated ADC data (values returned as volts, calibrate from cal file)\n");
	scanf("%d", &selection);
	getchar();

	switch(selection)
	{
		case 0:
			Sample_ReadADCsRAW(pPLXDev, commMode);
			break;

		case 1:
			Sample_ReadADCswCal(pPLXDev, commMode);
			break;
		default:
			printf("Invalid Selection.\n");
			break;
	}
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_6800_WriteDACs
//
//	Description: Write to DAC
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_6800_WriteDACs(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	UINT32 selection;

	printf("0 = Write raw DAC data (16 bit 2's complement)\n");
	printf("1 = Write calibrated DAC data (calibrated from cal file)\n");
	scanf("%d", &selection);
	getchar();

	switch(selection)
	{
		case 0:
			Sample_WriteDACsRAW(pPLXDev, commMode);
			break;

		case 1:
			Sample_WriteDACswCal(pPLXDev, commMode);
			break;
		default:
			printf("Invalid Selection.\n");
			break;
	}
}
///////////////////////////////////////////////////////////////////////////////
//	Sample_6800_RWDititalIO
//
//	Description: Digital I/O
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		T_6800ConfigParams pmod:
//			Pointer to 6800 Configuration Parameters.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////
void Sample_6800_RWDititalIO
(
	struct PLXDevice *pPLXDev,
	T_6800ConfigParams *pmod,
	UINT32 commMode
)
{
	UINT32 data[64];
	INT32 error;

	printf("In (1) or Out (0): ");
	scanf("%d", &data[1]);

	if(data[1])
		pmod->digitalIOPortDirection = 0x0;			
	else
		pmod->digitalIOPortDirection = 0x3;

//			calc68_CSR(pmod);		// Not used by 6800
	data[0] = pmod->digitalIOPortDirection;

	if(!data[1]) 
	{
		printf("Enter data (32 bits): 0x");
		scanf("%x", &data[1]);

		error = SI_Write
		(
			pPLXDev,  commMode, 1, 
			kDaughterAddr + (kDaughterOffsetCSR * kDSPBF), data		
		);	

		if (error != e_Err_NoError)
		{
			printf ("\nWRITE FAILED!\n");
			return;
		}

		if
		(
			write_DIO
			(
				pPLXDev, commMode, &data[1],
				kDaughterAddr, kDaughterOffsetDIO
			) != e_Err_NoError
		)
			printf("Write Error!\n");
	}
	else
	{
		error = SI_Write
		(
			pPLXDev,  commMode, 1, 
			kDaughterAddr + (kDaughterOffsetCSR * kDSPBF), data		
		);	

		if (error != e_Err_NoError)
		{
			printf ("\nWRITE FAILED!\n");
			return;
		}

		SI_Read
		(
			pPLXDev, commMode, 1,
			kDaughterAddr + (kDaughterOffsetDIO * kDSPBF), data
		);
		printf("DIO: 0x%x\n", data[0]);
	}
}


///////////////////////////////////////////////////////////////////////////////
//	Sample_6800_Xfere2PROMtoFPGA
//
//	Description: Transfer Cal PROM contents to Cal Table (SI-MOD66 only)
//
//	Inputs:
//		PPLXDevice pPLXDev:
//			Handle to device.
//		UINT32 commMode:
//			Communication Mode, must be Hostpoll by default.
//
//	Outputs:
//		None.
//
//	Rev. Notes:
//		2007-09-07 Ley.
//			Created.
//
///////////////////////////////////////////////////////////////////////////////

void Sample_6800_Xfere2PROMtoFPGA(struct PLXDevice *pPLXDev, UINT32 commMode)
{
	INT32 error;

	error = Xfere2PROMtoFPGA
	(
		pPLXDev,
		commMode
	);
	if(error)
	{
		printf("Error transfering contents\n");
		printf("from Cal PROM to Cal Table\n");
		printf("Error code = %d\n", error);
	}
}


///////////////////////////////////////////////////////////////////////////////
// Following functions are obsolete and not used by MOD6800.
// They remain here just as a sample.
//////////////////////////////////////////////////////////////////////////
// user can dynamically set the clocksrc
// Note: This function is not used by SI_Sample
// Note: Obsolete, no longer used but remain as a sample. ILey 2007_09_18.
//
//Obsolete, no longer used but remain as a sample. ILey 2007_09_18.
void set68_clksrc(T_6800ConfigParams *mod)
{
	UINT32 statusLatch;

	printf("Enter Clock Source (0=DDS0, 1=External, 2=EventCounter0): ");
	scanf("%d",&statusLatch);
	switch(statusLatch)
	{
	case 0:	//DDS
		mod->timingParams.sampleRateManagement.sampleClockSource = DDS0;
		break;
	case 1:	//External
		mod->timingParams.sampleRateManagement.sampleClockSource = External;
		break;
	case 2:	//Counter
	default:
		mod->timingParams.sampleRateManagement.sampleClockSource = EventCounter0;
		break;
	}

	// Here call a function similar to: "default_configure_MOD68(mod)",
	// but in which sampleClockSource will not be modified, but write
	// down to DSP all configuration params needed.

	getchar();	
}


//////////////////////////////////////////////////////////////////////////
// calculates the CSR register based on the mod struct
// passed in.
// Note: Obsolete, no longer used but remain as a sample. ILey 2007_09_18.
//
void calc68_CSR(T_6800ConfigParams *mod)
{
/*
	//clear csr
	mod->CSR = 0;
	//set all the bits
	mod->CSR |= mod->miscparams.DAC0_reset;
	mod->CSR |= mod->miscparams.DAC1_reset << 1;
	mod->CSR |= mod->miscparams.DAC2_reset << 6;
	mod->CSR |= mod->miscparams.DAC3_reset << 7;

	mod->CSR |= mod->miscparams.DDS_reset << 2;
	mod->CSR |= mod->miscparams.DIO_dir << 3;

	mod->CSR |= mod->miscparams.clksrc;
*/
}

