///////////////////////////////////////////////////////////////////////////////
//	mod68.h	: this is the filename for SI daughter module 6800 program header .
//
//	Description:
//		Contains the defines for the MOD 6800 such as register address offsets
//		Also has the enums and structs that make up the mod struct used
//		to setup and configure the MOD 6800 board.
//
//	Revision History:
//		2003-01-28: Colin
//			Created  
//
///////////////////////////////////////////////////////////////////////////////

// Note: These values are valid for 6800 only
// MOD68xx Timing_CSR[12:9] mapped at offset 0x201, bits ADC_SAMPLE_SRC[3:0]

enum clockSource 
{ 
	DDS0 = 0,		// ADC_SAMPLE_SRC[3:0] = 0, select DDS0
	DDS1,
	Ext_DDS,
	External, 
	EventCounter0,
	EventCounter1,	// ADC_SAMPLE_SRC[3:0] = 5, select EC1
	EventCounter2,
	EventCounter3,
	GPIO0,
	GPIO1,
	GPIO2,			// ADC_SAMPLE_SRC[3:0] = 10, select GPIO2
	GPIO3
};

#include "../common/common_simod.h"

//6800 Daughter Module defines

#define kDaughterNumMux			4
#define kDaughterMaxChannels	16
#define kDaughterSyncBit		8


#define kDSP68xxTimerCSR		0x201
#define kDSP68xxClearECCSR		0x202
#define kDSP68xxRegOffset		0x206
#define kDSP68xxIntDDS0PA		0x210
#define kDSP68xxIntDDS1PA		0x211
#define kDSP68xxEEPROMtoFPGACal	0x213
#define kDSP68xxExtDDSPA		0x218
#define kDSP68xxExtDDSLD		0x219
#define kDSP68xxDACCSR			0x220
#define kDSP68xxDIOCSR			0x224
#define kDSP68xxDIOData			0x225
#define kDSP68xxGPIOCSR			0x228
#define kDSP68xxAuxDIOCSR		0x22C
#define kDSP68xxPulseCSR		0x230
#define kDSP68xxQECSR			0x250
#define kDSP68xxScanTable		0x300

#define kDaughterOffsetDIO		kDSP68xxDIOData
#define kDaughterOffsetCSR		kDSP68xxDIOCSR

#define kMaxADCRate				250000
#define kDSP68xxNumberOfECs		4

#define kDaughterCSRddsResetBit	0x4
#define kDaughterCSRuPsBit		0x20

//#define kDDSOutSampleRate		2		// Not used by 6800

// These 3 variables must be in this file, but because of compiler
// problems, they're declared in common_simod.h
//#define kDaughterNumDAC			16
//#define kDaughterOffsetDAC		0x100
//#define kScaleFactor				(float) (20 / pow(2, 16));


// structures for Mod6800.
// For proper values for each parameter please refer to manual of Mod6800

typedef struct
{
	UINT32 channelsMux;
	Channel mux[kDaughterNumMux][kDaughterMaxChannels];
}T_6800ADCParams;

typedef struct
{
	UINT32 sampleRateDDS0;
	UINT32 sampleRateDDS1;
	UINT32 sampleRateExtDDS;
	UINT32 maxADCRate;
}T_6800SampleRateValues;

typedef struct
{
	UINT32 extClockEnable;
	UINT32 sampleMode;
	UINT32 dds0TriggerMode;
	UINT32 dds1TriggerMode;
	UINT32 sampleClockSource;
}T_6800SampleRateManagement;

typedef struct
{
	UINT32 ecReset;
	UINT32 ecEnable;
	UINT32 ecWrapMode;
	UINT32 ecPolarity;
	UINT32 ecTickClockSource;
	UINT32 ecThreshold;
}T_6800ECParams;

typedef struct
{
	T_6800SampleRateValues sampleRateValues;
	T_6800SampleRateManagement sampleRateManagement;
	T_6800ECParams ecParams[4];
}T_6800TimingParams;

typedef struct
{
	UINT32 gpioFunction1_0;
	UINT32 gpioFunction3_2;
	UINT32 freeCounterSource;
}T_6800GPIOConfig;

typedef struct
{
	UINT32 pulseIn0Reset;
	UINT32 pulseOut0Reset;
	UINT32 pulseIn1Reset;
	UINT32 pulseOut1Reset;
	UINT32 pulseOut0HighTickCount;
	UINT32 pulseOut0LowTickCount;
	UINT32 pulseOut1HighTickCount;
	UINT32 pulseOut1LowTickCount;
}T_6800PulseParams;

typedef struct
{
	UINT32 QE0DivideRatio;
	UINT32 QE0CountMode;
	UINT32 QE0IndexMode;
	UINT32 QE1DivideRatio;
	UINT32 QE1CountMode;
	UINT32 QE1IndexMode;
}T_6800QEParams;

typedef struct
{
	T_6800GPIOConfig gpIOConfig;
	UINT32 auxDIO;
	T_6800PulseParams pulseParams;
	T_6800QEParams qeParams;
}T_6800GPIOParams;

typedef struct
{
	T_6800ADCParams adcParams;
	T_6800TimingParams timingParams;
	T_6800GPIOParams gpIOParams;
	UINT32 dacClock;
	UINT32 digitalIOPortDirection;
}T_6800ConfigParams;


//6800 Daughter Module Menu define
#define kDMENU "\n\
===========================================================================\n\
6800 Daughter Module Options\n\n\
1  Set Default Configuration (Clocks, Muxes, etc.)\n\
2  Transfer Calibration data to Cal Table\n\
3  Read ADC Data\n\
4  Write to DAC\n\
5  Digital I/O\n\
\n0  Exit\n\
===========================================================================\n\
Enter Slection: "

//function headers
void default_configure_MOD68(T_6800ConfigParams *mod);

void calc68_IOParams
(
	T_6800ConfigParams *mod,
	UINT32 *adcSampleRates,
	UINT32 *channelParams,
	UINT32 *dacCSR,
	UINT32 *timingCSRs,
	UINT32 *gpioCSR,
	UINT32 *auxDIOCSR,
	UINT32 *qeCSR,
	UINT32 *pulseCSR,
	UINT32 *dioCSR
);

void calc68_CSR(T_6800ConfigParams *mod);

INT32 setup68_DaughterMod
(
	struct PLXDevice *pPLXDev,
	T_6800ConfigParams *mod,
	UINT32 commMode,
	UINT32 dspBusClkSpeed
);

void Sample_6800(struct PLXDevice *pPLXDev, UINT32 dspBusClkSpeed);

void set68_clksrc(T_6800ConfigParams *mod);

void Sample_6800_Config
(
	struct PLXDevice *pPLXDev, 
	T_6800ConfigParams *pmod,
	UINT32 commMode,
	UINT32 dspBusClkSpeed
);

void Sample_6800_CalConfig(struct PLXDevice *pPLXDev, UINT32 commMode);

void Sample_6800_ReadADCs(struct PLXDevice *pPLXDev, UINT32 commMode);

void Sample_6800_WriteDACs(struct PLXDevice *pPLXDev, UINT32 commMode);

void Sample_6800_RWDititalIO
(
	struct PLXDevice *pPLXDev,
	T_6800ConfigParams *pmod,
	UINT32 commMode
);

void Sample_6800_Xfere2PROMtoFPGA(struct PLXDevice *pPLXDev, UINT32 commMode);
