///////////////////////////////////////////////////////////////////////////////
//	sicommon_plxc33.cpp
//
//	Description:
//		Common hardware access routines for PLX with C33. 
//		These calls are PLX C33 specific.
//
//	Revision History:
//		2002-05-30: Av & Ley
//			Created
//		2002-06-25: mik
//			Changed comm mode to new method using bits.
//			Removed read/write CSR calls (not used).
//		2002-06-28: mik
//			Added choppping up of large count transfers for addon-init.
//			Fixed comments that were left over from C6711.
//		2002-07-01: mik
//			Changed all instances of SIHW_ to SI_
//		2002-07-08: mik
//			Fixed SI_PLXC33_ResetBoard to work with all modes.
//			Better error handling for CoffLoad.
//			Clean up comments.
//		2002-07-12: mik
//			Added timeout for BMDMA read/write.
//		2002-09-16: mik
//			Moved all pOverlapped and callback stuff into API.
///////////////////////////////////////////////////////////////////////////////

#include "string.h"
#include "SIQsicommon_plxc33.h"			// PLXC33 board specific

///////////////////////////////////////////////////////////////////////////////
// Combination of Device ID and Vendor ID for C33

#define kDevice_DIDVID_C33		0x101080bb
#define kDevice_DIDVIDMask		0x000f0000

///////////////////////////////////////////////////////////////////////////////
// function prototypes that are only used by this cpp file.

INT32 SI_PLXC33_ParseDSPCofffile
(
	PPLXDevice pPLXDev, UINT32 *dspMem, char *buffer, 
	UINT32 *size, UINT32 loadableSections, UINT32 *dspAddr,
	UINT32 **data, UINT32 &dspMemPtr
);

INT32 SI_PLXC33_ConvertDSPAddress
(
	UINT32 dspAddr,
	UINT32 *plxAddr
);

INT32 SI_PLXC33_DSPActiveComm1
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 srcAddr, UINT32 dstAddr, 
	UINT32 plxAddr, UINT32 *hostAddr
	
);

INT32 SI_PLXC33_DSPActiveComm2_Block
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm2_Point
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm3_SyncFlag
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm4_ReadWrite
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm4_WaitFlag
(
	PPLXDevice pPLXDev, 
	UINT32 flagValue, UINT32 timeoutMSec
);

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_OpenDriver
//
//	Description:
//		Attempts to link a valid Driver to our program. Extracts DID & VID
//		to assure it's 
//
//	Parameters:
//	
//
//

INT32 SI_PLXC33_OpenDriver
(	
	UINT32 board, 
	PPLXDevice pPLXDev
)
{
	INT32 error;
	UINT32 didVid[16];

	// 1. open the driver.
	error = SI_PLX_OpenDriver(board, pPLXDev);
	if (error != e_Err_NoError)
		return error;

	// 2. Check VID/DID
	error = SI_PLX_ReadPCI_ConfSpace( pPLXDev, 4, 0, didVid );
	if (error != e_Err_NoError)
	{
		SI_PLX_CloseDriver( pPLXDev );
		return error;
	}

	// mask out general purpose nibble
	didVid[0] &= ~kDevice_DIDVIDMask;
	if (didVid[0] != kDevice_DIDVID_C33)
	{
		SI_PLX_CloseDriver( pPLXDev );
		return e_Err_DeviceNotFound;
	}

	// 4. Get addon-init pointer and value
	error = SI_PLX_GetAddonBufferAddr( pPLXDev, PLX_DMRR );

	// we return regardless of addon init error.
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ResetBoard
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 resetMode: Toggle or keep in reset.

INT32 SI_PLXC33_ResetBoard
(
	PPLXDevice pPLXDev, 
	UINT32 resetMode	
)
{
	UINT32 tempData;
	INT32 error;
	
	switch(resetMode)
	{
	case e_Reset_Toggle:
	case e_Reset_Assert:
	case e_Reset_Deassert:
		break;

	default:
		return e_Err_ResetInvalidMode; 
	}

	if ( (resetMode == e_Reset_Toggle) || (resetMode == e_Reset_Assert) )
	{
		tempData = 0;

		error = 
			SI_PLXC33_WriteTarget( pPLXDev, 0, 1, kPLXC33_CSRAddr, &tempData );
		if( error != e_Err_NoError )
			return error;
	}

	if ( (resetMode == e_Reset_Toggle) || (resetMode == e_Reset_Deassert) )
	{
		tempData = 1;

		error = 
			SI_PLXC33_WriteTarget( pPLXDev, 0, 1, kPLXC33_CSRAddr, &tempData );
		if( error != e_Err_NoError )
			return error;

		// Wait for DSP to start running. May need time to init itself.
		SI_PLX_Sleep(500);
	
	}
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ParseDSPCofffile
//
//	Description:
//		Function to prepare the COFF information
//		to be written to the boot ram of the DSP
//		
//		Internal function only, used to load coff and verify.
//
///////////////////////////////////////////////////////////////////////////////
//	Format for wordsBeforeCoff, wordsAfterCoff; this example adds two sections.
//		word[0] = total count excluding this word. 
//				For this example, it's 5+N-1+M-1
//		word[1] = type
//		word[2] = count
//		word[3] = src (typically 0)
//		word[4] = dst (dsp addr where the following data will go
//		word[5] = data[0]
//		word[5+N-1] = data[N-1]
//		word[5+N] = type
//		word[5+N+1] = count
//		word[5+N+2] = src (typically 0)
//		word[5+N+3] = dst (dsp addr where the following data will go
//		word[5+N+4] = data[0]
//		word[5+N-1+M-1] = data[M-1]
//
//	Format for wordsAfterEnd;
//		word[0] = total count excluding this word. 
//				For this example, it's N
//		word[1] = data[0]
//		.
//		.
//		.
//		word[N] = data[N-1]

INT32 SI_PLXC33_ParseCoff
(
	PPLXDevice pPLXDev, char *filename, 
	UINT32 *dspMem, UINT32 dspMemCount, UINT32 *dspMemLoc, 
	UINT32 *wordsBeforeCoff, UINT32 *wordsAfterCoff, UINT32 *wordsAfterEnd
)
{
	INT32 error;
	UINT32 cnt, cnt1;
	UINT32 dspMemPtr, *ptr;

	dspMemPtr = 0x400;

	// for coff parsing
	UINT32
		dspAddr[kCOFF_MaxNumberOfSections], 
		size[kCOFF_MaxNumberOfSections], 
		*data[kCOFF_MaxNumberOfSections], 
		loadableSections;
	char *buffer;

	// first, zero out the dspMem array to remove residual garbage
	for(cnt=0; cnt<dspMemCount; cnt++)
		dspMem[cnt] = 0;

	// 2. Add extra params that loads before COFF
	ptr = wordsBeforeCoff;
	if (ptr != NULL)
	{
		if(dspMemPtr + ptr[0] >= dspMemCount)
			return e_Err_BufferTooSmall;

		for (cnt = 0; cnt < ptr[0]; cnt++)
			dspMem[dspMemPtr++] = ptr[1+cnt];	//data
	}

	// 3. parse the actual DSP coff file
	buffer = (char *)malloc(kCOFF_MaxBufferSize);
	if (buffer == NULL)
		return e_Err_WindowsMemError;

	// parse coff file to determine what should be loaded.
	error = 
		ParseCoff
		(
			filename, buffer, 
			dspAddr, size, data, &loadableSections, 0
		);
	if ( error != e_Err_NoError)
	{
		free(buffer);
		return error;

	}

	// load all sections from COFF file.
	for (cnt = 0; cnt < loadableSections; cnt++)
	{	
		// for C6x, size is returned as byte count, but the driver
		//	is expecting word count.
//		size[cnt] >>= 2;	// not for C33

		if ( dspAddr[cnt] == (kPLXC33_CoffLoadOffset) )
		{
			//	Inside of first 1k is copied blindly.
			for (cnt1 = 0; cnt1 < size[cnt]; cnt1++)
				dspMem[ cnt1 ] = data[cnt][cnt1];
		}
		else
		{
			//	Outside of first 1k of DSP code must be loaded with header.
			if(dspMemPtr + (4 + size[cnt]) >= dspMemCount)
			{
				free(buffer);
				free(dspMem);
				return e_Err_BufferTooSmall;
			}
			dspMem[dspMemPtr++] = 0;			// type
			dspMem[dspMemPtr++] = size[cnt];	// count
			dspMem[dspMemPtr++] = 0;			// src
			dspMem[dspMemPtr++] = dspAddr[cnt];	// dst
			for (cnt1=0; cnt1<size[cnt]; cnt1++)
				dspMem[dspMemPtr++] = data[cnt][cnt1];
		}
	}
	free(buffer);

	// 4. Add extra params that occurs after COFF
	ptr = wordsAfterCoff;
	if (ptr != NULL)
	{
		if(dspMemPtr + ptr[0] >= dspMemCount)
			return e_Err_BufferTooSmall;

		for (cnt=0; cnt<ptr[0]; cnt++)
			dspMem[dspMemPtr++] = ptr[1+cnt];	//data
	}

	// 5. write end marker
	if(dspMemPtr + 2 >= dspMemCount)
		return e_Err_BufferTooSmall;

	dspMem[dspMemPtr++] = 0xffffffff;
	dspMem[dspMemPtr++] = 0xffffffff;

	// 6. write stuff after end marker
	ptr = wordsAfterEnd;
	if (ptr != NULL)
	{
		if(dspMemPtr + ptr[0] >= dspMemCount)
			return e_Err_BufferTooSmall;

		for (cnt=0; cnt<ptr[0]; cnt++)
			dspMem[dspMemPtr++] = ptr[1+cnt];	//data
	}

	// 7. this many words are written to DSP
	*dspMemLoc = dspMemPtr;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_CoffLoad
//
//	Description:
//		Loads coff file. Does not check.
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		char *filename	: File to access.
//		UINT32 coffOptions: DSP Active or DSP Inactive.
//		UINT32 *wordsBeforeCoff: stuff that loads before actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterCoff: stuff that loads after actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterEnd: stuff that loads after end marker
//			see comments for SI_PLXC6711_ParseCoff

INT32 SI_PLXC33_CoffLoad
(
	PPLXDevice pPLXDev, 
	char *filename, UINT32 coffOptions, 
	UINT32 *wordsBeforeCoff, UINT32 *wordsAfterCoff, UINT32 *wordsAfterEnd
)
{
	INT32 error;
	UINT32	*dspMem, dspMemLoc, cnt;
	UINT32	dspMemCount = kCOFF_MaxBufferSize / 4;

	// stuff for DSP active coff loading
	UINT32
		dspAddr[kCOFF_MaxNumberOfSections], 
		size[kCOFF_MaxNumberOfSections], 
		*data[kCOFF_MaxNumberOfSections],
		loadableSections; 

	// make sure the driver is in block mode, in case it's in point mode
	//	by accident.
	error = 
		SI_PLX_DriverConfig_BlockPoint
		(
			pPLXDev, kFlagDriverConfig_Block
		);
	if (error != e_Err_NoError)
		return error;

	// malloc for dsp memory
	dspMem = (UINT32 *)malloc(kCOFF_MaxBufferSize);
	if (dspMem == NULL)
		return e_Err_WindowsMemError;

	//	DSP Active. Load using HOSTPOLL.
	if (coffOptions == e_Coffload_DSPActive)
	{
		// 1. parse coff file
		error = 
			ParseCoff
			(
				filename, (char *)dspMem, 
				dspAddr, size, data, &loadableSections, 0
			);
		if ( error != e_Err_NoError)
		{
			free(dspMem);
			return error;
		}
	
		// 2. load all sections from COFF file.
		for (cnt=0; cnt<loadableSections; cnt++)
		{	
			// for C3x size is returned as word count
			error = 
				SI_PLXC33_WriteHostpoll
				(
					pPLXDev, 
					size[cnt], 
					dspAddr[cnt], 
					data[cnt]
				);
			if( error != e_Err_NoError )
			{
				free(dspMem);
				return error;
			}
		}
		free(dspMem);
		return e_Err_NoError;
	}
	
	// DSP inactive loading.
	if (coffOptions == e_Coffload_DSPInactive)
	{
		// parse the coff file
		error = 
			SI_PLXC33_ParseCoff
			(
				pPLXDev, filename, dspMem, dspMemCount, &dspMemLoc, 
				wordsBeforeCoff, wordsAfterCoff, wordsAfterEnd
			);
		if ( error != e_Err_NoError)
		{
			free(dspMem);
			return error;
		}

		error = 
			SI_PLXC33_ResetBoard
			(
				pPLXDev, 
				e_Reset_Assert
			);
		if ( error != e_Err_NoError)
		{
			free(dspMem);
			return error;
		}

		// load all sections from COFF file, as a single write.
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev,
				0,
				dspMemLoc,
				kPLXC33_CoffLoadOffset,
				dspMem
			);
		if( error != e_Err_NoError )
		{
			free(dspMem);
			return error;
		}
		// write reset vector to CoffFileOffset so when reset is removed it
		// starts our si_c_int00 routine, parsing cofffile to correct locations
		cnt = kPLXC33_CoffLoadOffset;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev,
				0,
				1,
				kPLXC33_ResetVectorAddr,
				&cnt
			);
		if( error != e_Err_NoError )
		{
			free(dspMem);
			return error;
		}
		free(dspMem);
		return e_Err_NoError;
	}		
	free(dspMem);
	return e_Err_UnknownCommand;

}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_CoffVerify
//
//	Description:
//		Checks the coff file with what's in DSP memory. Does not start the DSP.
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		char *filename	: File to access.
//		UINT32 coffOptions: Target or hostpoll
//		UINT32 *extrasBeforeCoff: stuff that gets loaded before actual coff
//		UINT32 *extrasAfterCoff: stuff that gets loaded after actual coff
//			format for extras is type, count, src, dst, data0, ..., dataN-1
//		UINT32 *wordsBeforeCoff: stuff that loads before actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterCoff: stuff that loads after actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterEnd: stuff that loads after end marker
//			see comments for SI_PLXC6711_ParseCoff

INT32 SI_PLXC33_CoffVerify
(
	PPLXDevice pPLXDev, 
	char *filename, UINT32 coffOptions, 
	UINT32 *wordsBeforeCoff, UINT32 *wordsAfterCoff, UINT32 *wordsAfterEnd
)
{
	INT32 error;
	
	UINT32	*dspMem, dspMemLoc, 
			dspMemCount = kCOFF_MaxBufferSize / 4;
	UINT32	*verify, cnt, cnt1;

	// stuff for DSP active coff loading
	UINT32
		dspAddr[kCOFF_MaxNumberOfSections], 
		size[kCOFF_MaxNumberOfSections], 
		*data[kCOFF_MaxNumberOfSections], 
		loadableSections;

	// make sure the driver is in block mode, in case it's in point mode
	//	by accident.
	error = 
		SI_PLX_DriverConfig_BlockPoint
		(
			pPLXDev, kFlagDriverConfig_Block
		);
	if (error != e_Err_NoError)
		return error;

	// malloc for dsp memory
	dspMem = (UINT32 *)malloc(kCOFF_MaxBufferSize);
	if (dspMem == NULL)
		return e_Err_WindowsMemError;

	//	DSP Active. Load using HOSTPOLL.
	if (coffOptions == e_Coffload_DSPActive)
	{
		// 1. parse coff file
		error = 
			ParseCoff
			(
				filename, (char *)dspMem, 
				dspAddr, size, data, &loadableSections, 0
			);
		if ( error != e_Err_NoError)
		{
			free(dspMem);
			return error;
		}
	
		// allocate verify mem only after parsing coff to reduce memory load.
		verify = (UINT32 *)malloc(kCOFF_MaxBufferSize);
		if (verify == NULL)
		{
			free(dspMem);
			return e_Err_WindowsMemError;
		}

		// 2. load all sections from COFF file.
		for (cnt=0; cnt<loadableSections; cnt++)
		{	
			error = 
				SI_PLXC33_ReadHostpoll
				(
					pPLXDev, 
					size[cnt], 
					dspAddr[cnt], 
					verify
				);
			if( error != e_Err_NoError )
			{
				free(dspMem);
				free(verify);
				return error;
			}

			// compare
			for (cnt1=0; cnt1 < size[cnt]; cnt1++)
				if ( data[cnt][cnt1] != verify[cnt1] )
				{
					free(dspMem);
					free(verify);
					return e_Err_CompareError;
				}
		}

		free(dspMem);
		free(verify);
		return e_Err_NoError;
	}

		// DSP inactive loading.
	if (coffOptions == e_Coffload_DSPInactive)
	{
		error =
			SI_PLXC33_ParseCoff
			(
				pPLXDev, filename, dspMem, dspMemCount, &dspMemLoc, 
				wordsBeforeCoff, wordsAfterCoff, wordsAfterEnd
			);

		if ( error != e_Err_NoError)
		{
			free(dspMem);
			return error;
		}

		// allocate verify mem only after parsing coff to reduce memory load.
		verify = (UINT32 *)malloc(kCOFF_MaxBufferSize);
		if (verify == NULL)
		{
			free(dspMem);
			return e_Err_WindowsMemError;
		}

		// 4. read from boot ram
		error = 
			SI_PLXC33_ReadTarget
			(
				pPLXDev,
				0,
				dspMemLoc,
				kPLXC33_CoffLoadOffset,
				verify
			);
		if( error != e_Err_NoError )
		{
			free(dspMem);
			free(verify);
			return error;
		}
			
		// 5. compare
		for (cnt = 0; cnt < dspMemLoc; cnt++)
		{
			if ( dspMem[cnt] != verify[cnt] )
			{
				free(dspMem);
				free(verify);
				return e_Err_CompareError;
			}
		}

		error = 
			SI_PLXC33_ReadTarget
			(
				pPLXDev,
				0,
				1,
				kPLXC33_ResetVectorAddr,
				verify
			);
		if( error != e_Err_NoError )
		{
			free(dspMem);
			free(verify);
			return error;
		}

		// compare
		if ( verify[0] != kPLXC33_CoffLoadOffset )
		{
			free(dspMem);
			free(verify);
			return e_Err_CompareError;
		}

		free(dspMem);
		free(verify);
		return e_Err_NoError;
	}
	
	free(dspMem);
	return e_Err_UnknownCommand;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ConvertDSPAddress
//
//	Description:
//		Converts input DSP address to PLX address for target and BM DMA.
//		C33 addresses as seen by PLX. All these are DWORD addresses.
//		These are write only. When written with 1, they toggle for 2 ECLK.
//		The Interrupt DSP address is obtained as follows -->
//			FF0001 can be decoded to -->	
//		******************************************************
//		* B23 | B22 | B21 | B20 | B19 | B18 | ......... | B0 *
//		******************************************************
//			1111 1111 0000 0000 0000 0001
//
//		Now on the plx side the upper 2 bits are page bits and
//		bits 21 to 19 are ignored 
//		******************************************************
//		* B20 | B19 | B18 | ............................| B0 *
//		******************************************************
//		so pg = 11 = 3
//		so we are left with bits 21 -> 0
//			11 1111 0000 0000 0000 0001 
//		After bits 21 -> 19 are removed,we have
//			111 0000 0000 0000 0001 
//
//		Appending the page bits
//			1 1111 0000 0000 0000 0001 
//		which is Hex
//			0x1F0001
//		translating from DWORD boundary to BYTE boundary(left shift by 2)
//			1 1111 0000 0000 0000 000100 
//		rearraging the bits
//			0111 1100 0000 0000 0000 0100 
//		which is Hex 
//			0x7C0004
//		Which is the Address used by PLX PassThrough Write to cause DSP 
//		Interrupt. The above Process also applies to CSR for DSP Reset
//		(0xFF0000)
//
//	Parameters:
//		UINT32 dspAddr	: DSP address from application.
//		UINT32 *plxAddr	: PLX address to use in Target and BM DMA transfers.

INT32 SI_PLXC33_ConvertDSPAddress
(
	UINT32 dspAddr,
	UINT32 *plxAddr
)
{
	UINT32 page;

	// DSP page
	page = (dspAddr & 0xC00000) >> 3;
	*plxAddr = (page + (dspAddr & 0x07ffff)) << 2;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadTarget
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		region			: PLX space
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_ReadTarget
(
	PPLXDevice pPLXDev, 
	UINT32 region, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{

	INT32 error;
	UINT32 plxAddr;

	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;

	error = SI_PLX_ReadTarget( pPLXDev, region, count, plxAddr, hostAddr );
	if ( error != e_Err_NoError)
		return error;

			
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteTarget
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		region			: PLX space
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_WriteTarget
(
	PPLXDevice pPLXDev, 
	UINT32 region, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxAddr;

	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;

	error = SI_PLX_WriteTarget( pPLXDev, 0, count, plxAddr, hostAddr );
	if ( error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteBlockDMA
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_WriteBlockDMA
(
	PPLXDevice pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxAddr;
	UINT32 timeout;

	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;

	// set timer (for long transfers)
	timeout = count / 1000 + 1000;	// 1000 word / sec + 1 sec

	error = SI_PLX_SetTimeout( pPLXDev, timeout );
	if (error != e_Err_NoError)
		return error;
	
	error = SI_PLX_WriteBlockDMA( pPLXDev, count, plxAddr, hostAddr );
	if ( error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadBlockDMA
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_ReadBlockDMA
(
	PPLXDevice pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxAddr;
	UINT32 timeout;

	// convert addr to DSP
	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;
	
	// set timer (for long transfers)
	timeout = count / 1000 + 1000;	// 1000 word / sec + 1 sec

	error = SI_PLX_SetTimeout( pPLXDev, timeout );
	if (error != e_Err_NoError)
		return error;

	error = 
		SI_PLX_ReadBlockDMA
		(
			pPLXDev, 
			count, 
			plxAddr,
			hostAddr
		);
	if ( error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_TranslateCommModeIndex
//
//	Description:
//		Converts commModeIndex (consecutive numbers) to actual commMode bits
//		that the DSP can use. Also returns the "name" of the corresponding
//		comm mode.
//		
//	Parameters:
//		UINT32 commModeIndex 	: CommMode index (consecutive number)
//		UINT32 *commMode		: Return value. Converted from comm Index.
//		UINT32 maxStringCount	: Max number of bytes allocated in String.
//		char *commName			: Pointer to string where the name will reside.

INT32 SI_PLXC33_TranslateCommModeIndex
(
	UINT32 commModeIndex, UINT32 *commMode, 
	UINT32 maxStringCount, char *commName

)
{
	UINT32 commMode1;
	char *stringVal = NULL;
	
	switch (commModeIndex)
	{

	case e_CommModeIdx_Addon_Async_IO_Int:
		commMode1 =	kCommModeMask_HostAddonInit 
			|	kCommModeMask_InterruptHost;
		stringVal = "Add-on: DSP Async IO - Host Int.";
		break;

	case e_CommModeIdx_Addon_Async_DMA_Int:
		commMode1 =	kCommModeMask_HostAddonInit 
			|	kCommModeMask_InterruptHost
			|	kCommModeMask_DMA;
		stringVal = "Add-on: DSP Async DMA - Host Int.";
		break;

	case e_CommModeIdx_Point_Sync_IO_Target_CommReg:
		commMode1 =	kCommModeMask_PointTransfer
			|	kCommModeMask_Sync;
		stringVal = "Target: DSP Sync IO CommReg";
		break;

	default:
		return e_Err_UnknownCommand;
	}

	*commMode = commMode1;

	// copy string name if there's room
	if ( (strlen (stringVal) + 1) < maxStringCount)
	{
		while ( (*stringVal) != 0 )
			*commName++ = *stringVal++;

		*commName++ = 0;
	}

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadHostpoll
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_ReadHostpoll
(
	PPLXDevice pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	return 
		SI_PLXC33_ReadDSPActive
		(
			pPLXDev, 
			e_CommModeIdx_Point_Sync_IO_Target_CommReg, 
			count, dspAddr, hostAddr
		);
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteHostpoll
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_WriteHostpoll
(
	PPLXDevice pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	return 
		SI_PLXC33_WriteDSPActive
		(
			pPLXDev, 
			e_CommModeIdx_Point_Sync_IO_Target_CommReg, 
			count, dspAddr, hostAddr
		);
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadDSPActive
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commModeIndex: Transfer mode as modeIndex (see enum).

INT32 SI_PLXC33_ReadDSPActive
(
	PPLXDevice pPLXDev, 
	UINT32 commModeIndex, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 srcAddr, dstAddr, plxAddr, commMode;

	if (count == 0)
		return e_Err_NoError;

	error = 
		SI_PLXC33_TranslateCommModeIndex
		(
			commModeIndex, &commMode, 0, NULL
		);
	if ( error != e_Err_NoError )
		return error;

	// DSP write, host read.
	commMode &= ~kCommModeMask_DSPRead;

	// reading from DSP means srcAddr is the addr we want to read.
	srcAddr = dspAddr;

	// Depending on commMode, determine what should be done.
	if (commMode & kCommModeMask_PointTransfer)
	{
		// hostpoll: sync IO
		dstAddr = kPLXC33_CommReg_Data;
		plxAddr = kPLXC33_CommReg_Data;
	}
	else
	{
		// block transfer. only for addoninit
		dstAddr = kPLXC33_CommReg_HostAddr;
		plxAddr = 0;
	}

	error = 
		SI_PLXC33_DSPActiveComm1
		(
			pPLXDev, 
			commMode, count, srcAddr, dstAddr, 
			plxAddr, hostAddr
		);
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteDSPActive
//
//	Description:
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commModeIndex: Transfer mode as modeIndex (see enum).

INT32 SI_PLXC33_WriteDSPActive
(
	PPLXDevice pPLXDev, 
	UINT32 commModeIndex, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 srcAddr, dstAddr, plxAddr, commMode;

	if (count == 0)
		return e_Err_NoError;

	error = 
		SI_PLXC33_TranslateCommModeIndex
		(
			commModeIndex, &commMode, 0, NULL
		);
	if ( error != e_Err_NoError )
		return error;

	// DSP read, host write
	commMode |= kCommModeMask_DSPRead;

	// write to DSP means dstAddr is the addr we want to write.
	dstAddr = dspAddr;

	// Depending on commMode, determine what should be done.
	if (commMode & kCommModeMask_PointTransfer)
	{
		// hostpoll: sync IO
		srcAddr = kPLXC33_CommReg_Data;
		plxAddr = kPLXC33_CommReg_Data;
	}
	else
	{
		// block transfer. only for addoninit
		srcAddr = kPLXC33_CommReg_HostAddr;
		plxAddr = 0;
	}

	error = 
		SI_PLXC33_DSPActiveComm1
		(
			pPLXDev, 
			commMode, count, srcAddr, dstAddr, 
			plxAddr, hostAddr
		);
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm1
//
//	Description:
//		Root level of DSPActiveComm function: DSP communications when DSP is 
//		active. Note that modes must be carefully selected or the board will 
//		lock.
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 commMode	: Transfer mode. Use OR of the following:
//			kCommModeMask_InterruptHost 
//			kCommModeMask_PointTransfer
//			kCommModeMask_DSPRead 
//			kCommModeMask_Sync 
//			kCommModeMask_DMA
//			kCommModeMask_HostAddonInit 
//			kCommModeMask_HostBMDMA
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 srcAddr	: DSP source address.
//		UINT32 dstAddr	: DSP destination address.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.


INT32 SI_PLXC33_DSPActiveComm1
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 srcAddr, UINT32 dstAddr, 
	UINT32 plxAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxCommAddr, interruptCSRValue, wordCount, cnt;

	// get addr to commreg as PLX will see them.
	error = 
		SI_PLXC33_ConvertDSPAddress
		(
			kPLXC33_CommReg,
			&plxCommAddr
		);
	if ( error != e_Err_NoError)
		return error;

	if (commMode & kCommModeMask_PointTransfer)
	{
		// point transfer cannot be the following
		if 
		(
			commMode & 
			(
					kCommModeMask_HostAddonInit
				|	kCommModeMask_DMA
				|	kCommModeMask_HostBMDMA
			)
		)
			return e_Err_UnknownCommand;

		// point must be sync
		if ( !(commMode & kCommModeMask_Sync) )
			return e_Err_UnknownCommand;

		// Point transfer is always assumed to be to/from CommReg.

		// setup the DSP
		error = 
			SI_PLX_SetupAddonInitParams
			(
				pPLXDev, 
				plxCommAddr, 
				commMode, count, srcAddr, dstAddr
			);
		if ( error != e_Err_NoError )
			return error;

		// now actually generate interrupt
		interruptCSRValue = kPLXC33_IntDSPValue;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_IntDSPAddr, &interruptCSRValue
			);
		if ( error != e_Err_NoError )
			return error;

		// do actual transfer.
		error = 
			SI_PLXC33_DSPActiveComm2_Point
			(
				pPLXDev, 
				commMode, count, plxAddr, hostAddr
			);

		return error;
	}

	// block transfer is always to addoninit memory.
	if ( !(commMode & kCommModeMask_HostAddonInit) )
		return e_Err_UnknownCommand;

	// block transfer can never be synchronized to flag.
	//	Using flag could cause lockup due to race condition.
	if (commMode & kCommModeMask_Sync)
		return e_Err_UnknownCommand;

	// block mode must be done through host interrupt. otherwise, 
	//	it can lock.
	if ( !(commMode & kCommModeMask_InterruptHost) )
		return e_Err_UnknownCommand;
	 
	if (pPLXDev->addonInitBufferSize == 0)
		return e_Err_CountTooBig;

	if ( count > pPLXDev->addonInitBufferSize )
	{
		// chop up the transfer into multiple since it won't fit in one.
		wordCount = count;
		for (cnt=0; cnt<(wordCount / pPLXDev->addonInitBufferSize); cnt++)
		{
			// setup the DSP
			error = 
				SI_PLX_SetupAddonInitParams
				(
					pPLXDev, 
					plxCommAddr, 
					commMode, pPLXDev->addonInitBufferSize, srcAddr, dstAddr
				);
			if ( error != e_Err_NoError )
				return error;

			// do actual transfer. 
			error = 
				SI_PLXC33_DSPActiveComm2_Block
				(
					pPLXDev, 
					commMode, pPLXDev->addonInitBufferSize, plxAddr, hostAddr
				);
			if (error != e_Err_NoError)
				return error;

			count -= pPLXDev->addonInitBufferSize;
			hostAddr += pPLXDev->addonInitBufferSize;
			if (commMode & kCommModeMask_DSPRead)
			{
				// DSP read, host write, change dst addr
				dstAddr += pPLXDev->addonInitBufferSize;
			}
			else
			{
				// DSP write, host read, change srcaddr
				srcAddr += pPLXDev->addonInitBufferSize;
			}
		}	// end for many buffers
	}	// end if larger than pPLXDev->addonInitBufferSize

	// make sure this check is done after large count addoninit.
	if (count == 0)
		return e_Err_NoError;

	// setup the DSP
	error = 
		SI_PLX_SetupAddonInitParams
		(
			pPLXDev, 
			plxCommAddr, 
			commMode, count, srcAddr, dstAddr
		);
	if ( error != e_Err_NoError )
		return error;

	// do normal block transfer. 
	error = 
		SI_PLXC33_DSPActiveComm2_Block
		(
			pPLXDev, 
			commMode, count, plxAddr, hostAddr
		);
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm2_Point
//
//	Description:
//		Level 0 of DSPActiveComm function.
//		Transfers using point of DSP, such as FIFO or commreg.
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commMode	: Transfer mode.

INT32 SI_PLXC33_DSPActiveComm2_Point
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 cnt, flag;

	// now read/write data using flag.

	//	DSP is using IO and synchronizing with flag. 

	// Make sure DSP got the parameters ok.
	// wait for flag
	error = 
		SI_PLXC33_DSPActiveComm4_WaitFlag
		(
			pPLXDev, 
			kCommModeFlag_ValidCommand, kCommModeTimeoutValue
		);
	if ( error != e_Err_NoError )
		return error;

	// read from DSP: set flag so that read data is ready.
	if ( !(commMode & kCommModeMask_DSPRead) )
	{
		// flag DSP that host is ready for first word to read
		flag = kCommModeFlag_HostReady;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_CommReg_Flag, &flag
			);
		if ( error != e_Err_NoError )
			return error;
	}	// end if for reading first word from DSP.

	// do the rest of the transfers.
	for (cnt=0; cnt<count; cnt++)
	{
		if ( !(commMode & kCommModeMask_DSPRead) && (cnt == count-1) )
		{
			// if this is last DSPread (host write), 
			//	don't wait for flag.
		}
		else
		{
			// wait for flag
			error = 
				SI_PLXC33_DSPActiveComm4_WaitFlag
				(
					pPLXDev, 
					kCommModeFlag_ValidCommand, kCommModeTimeoutValue
				);
			if ( error != e_Err_NoError )
				return error;
		}

		// if host addr is not NULL, then do transfer. only the point transfer 
		//	is valid and block transfer must have hostAddr set to NULL.
		if (hostAddr != NULL)
		{
			// Transfer one word just before setting the flag. 
			//	This should be FIFO or commreg_data location. count must be 1.
			error = 
				SI_PLXC33_DSPActiveComm4_ReadWrite
				(
					pPLXDev, 
					commMode, 1, plxAddr, &hostAddr[cnt]
				);
			if ( error != e_Err_NoError )
				return error;
		}

		// if this is read from DSP, and it's the last transfer, don't set flag
		if ( !(commMode & kCommModeMask_DSPRead) && (cnt == count-1) )
			break;

		// flag DSP that host is ready for another word
		flag = kCommModeFlag_HostReady;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_CommReg_Flag, &flag
			);
		if ( error != e_Err_NoError )
			return error;

	}	// end for

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm2_Block
//
//	Description:
//		Level 0 of DSPActiveComm function.
//		Transfers using block of DSP.
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commMode	: Transfer mode.

INT32 SI_PLXC33_DSPActiveComm2_Block
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
)
{
	UINT32 plxIntAddr, intValue;
	INT32 error;

	error = SI_PLXC33_ConvertDSPAddress(kPLXC33_IntDSPAddr, &plxIntAddr);
	if (error != e_Err_NoError)
		return error;

	intValue = kPLXC33_IntDSPValue;

	// this always assumes addon-init, host interrupt, no sync per word.
	if (commMode & kCommModeMask_DSPRead)	// host writes
	{
		return
			SI_PLX_WriteAddonInit
			(
				pPLXDev, 
				plxIntAddr, intValue, 
				count, hostAddr
			);
	}
	else
	{
		return
			SI_PLX_ReadAddonInit
			(
				pPLXDev, 
				plxIntAddr, intValue, 
				count, hostAddr
			);
	}

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm4_ReadWrite
//
//	Description:
//		Level 2 of DSPActiveComm function.
//		Depending on commMode, read/write using target/BMDMA.
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commMode	: Mode of read/write, target/BMDMA.

INT32 SI_PLXC33_DSPActiveComm4_ReadWrite
(
	PPLXDevice pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
)
{
	INT32 error;

	// read from DSP. This should be FIFO location.
	if (commMode & kCommModeMask_DSPRead)	// host writes
	{
		if (commMode & kCommModeMask_HostBMDMA)
			error = 
				SI_PLXC33_WriteBlockDMA
				(
					pPLXDev, 
					count, plxAddr, hostAddr
				);
		else
			error = 
				SI_PLXC33_WriteTarget
				(
					pPLXDev, 
					0, count, plxAddr, hostAddr
				);
	}
	else								// host reads
	{
		if (commMode & kCommModeMask_HostBMDMA)
			error = 
				SI_PLXC33_ReadBlockDMA
				(
					pPLXDev, 
					count, plxAddr, hostAddr
				);
		else
			error = 
				SI_PLXC33_ReadTarget
				(
					pPLXDev, 
					0, count, plxAddr, hostAddr
				);
	}	// end if DSP read

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm4_WaitFlag
//
//	Description:
//		Level 4 of DSPActiveComm function.
//		Waits for flag value until timout.
//
//	Parameters:
//		PPLXDevice pPLXDev	: Handle to the device driver.
//		UINT32 flagValue: Flag to wait for.
//		UINT32 timeoutMSec: Timout value in msec.

INT32 SI_PLXC33_DSPActiveComm4_WaitFlag
(
	PPLXDevice pPLXDev, 
	UINT32 flagValue, UINT32 timeoutMSec
)
{
	INT32 error;
	UINT32 cnt, flag, sleepTime;

	sleepTime = 100;	// time to sleep between successive checks.

	// wait for flag
	for (cnt=0; 1; cnt++)
	{
		error = 
			SI_PLXC33_ReadTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_CommReg_Flag, &flag
			);
		if ( error != e_Err_NoError )
			return error;

		if ( flag == flagValue )
			break;	// valid flag received

		SI_PLX_Sleep(sleepTime);					// give some time

		if ( ( cnt * sleepTime ) > timeoutMSec )
			return e_Err_AddonInitTimeoutError;	// error.
	}

	return e_Err_NoError;
}

