///////////////////////////////////////////////////////////////////////////////
//	sicommon_plxc33.cpp
//
//	Sheldon Instruments, Inc.
//
//	Description:
//		Common hardware access routines for PLX with C33. 
//		These calls are PLX C33 specific.
//
//	Functions:
//		SI_PLXC33_OpenDriver
//		SI_PLXC33_ResetBoard
//		SI_PLXC33_ParseCoff
//		SI_PLXC33_CoffWrite
//		SI_PLXC33_CoffVerify
//		SI_PLXC33_ConvertDSPAddress
//		SI_PLXC33_ReadTarget
//		SI_PLXC33_WriteTarget
//		SI_PLXC33_WriteBlockDMA
//		SI_PLXC33_ReadBlockDMA
//		SI_PLXC33_TranslateCommModeIndex
//		SI_PLXC33_ReadHostpoll
//		SI_PLXC33_WriteHostpoll
//		SI_PLXC33_ReadDSPActive
//		SI_PLXC33_WriteDSPActive
//		SI_PLXC33_DSPActiveComm1
//		SI_PLXC33_DSPActiveComm2_Point
//		SI_PLXC33_DSPActiveComm2_Block
//		SI_PLXC33_DSPActiveComm4_ReadWrite
//		SI_PLXC33_DSPActiveComm4_WaitFlag
//		SI_PLXC33_CoffLoad_GetMemoryParams
//		SI_PLXC33_RecvMessage
//		SI_PLXC33_SendMessage
//		SI_PLXC33_CommInterrupt
//
//	Revision History:
//		2002-05-30: Av & Ley
//			Created
//		2002-06-25: mik
//			Changed comm mode to new method using bits.
//			Removed read/write CSR calls (not used).
//		2002-06-28: mik
//			Added choppping up of large count transfers for addon-init.
//			Fixed comments that were left over from C6711.
//		2002-07-01: mik
//			Changed all instances of SIHW_ to SI_
//		2002-07-08: mik
//			Fixed SI_PLXC33_ResetBoard to work with all modes.
//			Better error handling for CoffLoad.
//			Clean up comments.
//		2002-07-12: mik
//			Added timeout for BMDMA read/write.
//		2002-09-16: mik
//			Moved all pOverlapped and callback stuff into API.
//		2006-10-16: Ley
//			Expanded GetBoardInfo so now it also outputs
//			DSP Bus Speed and Boundary Factor.
//			For SISample related programs: Added read NVRam to have the DSP
//			Bus speed in order to calculate parameters for DDS
//			for Sampling Rate.
//		2007-07-13: Ley
//			Updated C33 for messaging, except for SetXferSize which
//			it doesn't use.
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sicommon_plxc33.h"			// PLXC33 board specific
#include "../../../../../../common_ddk/os_lnx.h"

///////////////////////////////////////////////////////////////////////////////
// function prototypes that are only used by this cpp file.

INT32 SI_PLXC33_ConvertDSPAddress
(
	UINT32 dspAddr,
	UINT32 *plxAddr
);

INT32 SI_PLXC33_DSPActiveComm1
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 srcAddr, UINT32 dstAddr, 
	UINT32 plxAddr, UINT32 *hostAddr
	
);

INT32 SI_PLXC33_DSPActiveComm2_Block
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm2_Point
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm3_SyncFlag
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm4_ReadWrite
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
);

INT32 SI_PLXC33_DSPActiveComm4_WaitFlag
(
	struct PLXDevice *pPLXDev, 
	UINT32 flagValue, UINT32 timeoutMSec
);

//extern INT32 OS_Popup_ErrorMessageBox(char *text, char *caption, int errorCode, UINT32 type);

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_OpenDriver
//
//	Description:
//		Attempts to link a valid Driver to our program. Extracts DID & VID
//		to assure it's 
//
//	Parameters:
//	
//
//

INT32 SI_PLXC33_OpenDriver
(	
	struct SI_BoardInfo *pBoardInfo, 
	struct PLXDevice *pPLXDev
)
{
	INT32 error;
	UINT32 data;

	// 1. open the driver.
	error = SI_PLX_OpenDriver(pBoardInfo->boardID, pPLXDev);
	if (error != e_Err_NoError)
		return error;

	// 2. Check VID/DID
	error = SI_PLX_ReadPCI_ConfSpace( pPLXDev, 4, 0, pBoardInfo->didVid );
	if (error != e_Err_NoError)
	{
		SI_PLX_CloseDriver( pPLXDev );
		return error;
	}

	switch(pBoardInfo->didVid[0])
	{
		case kDevice_DIDVID_C33_XL:		// 1010
			pBoardInfo->cgHostBus = PCIPLX9054;
			pBoardInfo->dspType = C33;
			pBoardInfo->dspBusClkIndex = _75MHz;
			pBoardInfo->dspCoreClkSpeed = 75000000;
			pBoardInfo->dspBusClkSpeed = 75000000;
			pBoardInfo->cgDSP_DataSize = kPLXC33_DataSize;
			pBoardInfo->cgDSP_Bank0 = 0;
			pBoardInfo->cgDSP_Bank1 = 0;
			// 4. Note: Model has FPGA EEPROM,
			// don't check for FPGA bit loaded.
			break;
		case kDevice_DIDVID_C33_50e:	// 1011
		case kDevice_DIDVID_C33:		// 1012
			pBoardInfo->cgHostBus = PCIPLX9054;
			pBoardInfo->dspType = C33;
			pBoardInfo->dspBusClkIndex = _75MHz;
			pBoardInfo->dspCoreClkSpeed = 75000000;
			pBoardInfo->dspBusClkSpeed = 75000000;
			pBoardInfo->cgDSP_DataSize = kPLXC33_DataSize;
			pBoardInfo->cgDSP_Bank0 = 0;
			pBoardInfo->cgDSP_Bank1 = 0;

			// 4. Make sure the board is bit loaded
			// for this model only.
			error = 
				SI_PLX_ReadPCI_OpReg
				(
					pPLXDev, 1, PLX_CNTRL << 2, &data
				);
			if (error != e_Err_NoError)
			{
				SI_PLX_CloseDriver( pPLXDev );
				return error;
			}
			if(!(data & kMaskFpgaDONE))
			{
				SI_PLX_CloseDriver( pPLXDev );
				OS_Popup_ErrorMessageBox
				(
					"FPGA not loaded, please load FPGA\nbefore running application again",
					"Open Driver error",
					e_Err_FPGA_DoneNotHigh,
					0
				);
				return e_Err_FPGA_DoneNotHigh;
			}

			break;

		default:
			SI_PLX_CloseDriver( pPLXDev );
			return e_Err_DeviceNotFound;
			break;
	}

	// mask out general purpose nibble
	// Note: cannot find its use, but remained here as legacy just in case.
	pBoardInfo->didVid[0] &= ~kDevice_DIDVIDMask;

	// 4. Get addon-init pointer and value
	error = SI_PLX_GetAddonBufferAddr( pPLXDev, PLX_DMRR );

	// we return regardless of addon init error.
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ResetBoard
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 resetMode: Toggle or keep in reset.

INT32 SI_PLXC33_ResetBoard
(
	struct PLXDevice *pPLXDev, 
	UINT32 resetMode	
)
{
	UINT32 tempData;
	INT32 error;
	
	switch(resetMode)
	{
	case e_Reset_Toggle:
	case e_Reset_Assert:
	case e_Reset_Deassert:
		break;

	default:
		return e_Err_ResetInvalidMode; 
	}

	if ( (resetMode == e_Reset_Toggle) || (resetMode == e_Reset_Assert) )
	{
		tempData = 0;

		error = 
			SI_PLXC33_WriteTarget( pPLXDev, 0, 1, kPLXC33_CSRAddr, &tempData );
		if( error != e_Err_NoError )
			return error;
	}

	if ( (resetMode == e_Reset_Toggle) || (resetMode == e_Reset_Deassert) )
	{
		tempData = 1;

		error = 
			SI_PLXC33_WriteTarget( pPLXDev, 0, 1, kPLXC33_CSRAddr, &tempData );
		if( error != e_Err_NoError )
			return error;

		// Wait for DSP to start running. May need time to init itself.
		SI_PLX_Sleep(500);
	
	}
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ParseDSPCofffile
//
//	Description:
//		Function to prepare the COFF information
//		to be written to the boot ram of the DSP
//		
//		Internal function only, used to load coff and verify.
//
///////////////////////////////////////////////////////////////////////////////
//	Format for wordsBeforeCoff, wordsAfterCoff; this example adds two sections.
//		word[0] = total count excluding this word. 
//				For this example, it's 5+N-1+M-1
//		word[1] = type
//		word[2] = count
//		word[3] = src (typically 0)
//		word[4] = dst (dsp addr where the following data will go
//		word[5] = data[0]
//		word[5+N-1] = data[N-1]
//		word[5+N] = type
//		word[5+N+1] = count
//		word[5+N+2] = src (typically 0)
//		word[5+N+3] = dst (dsp addr where the following data will go
//		word[5+N+4] = data[0]
//		word[5+N-1+M-1] = data[M-1]
//
//	Format for wordsAfterEnd;
//		word[0] = total count excluding this word. 
//				For this example, it's N
//		word[1] = data[0]
//		.
//		.
//		.
//		word[N] = data[N-1]

/*
INT32 SI_PLXC33_ParseCoff
(
	struct PLXDevice *pPLXDev, char *filename, 
	UINT32 *dspMem, UINT32 dspMemCount, UINT32 *dspMemLoc, 
	UINT32 *wordsBeforeCoff, UINT32 *wordsAfterCoff, UINT32 *wordsAfterEnd
)
*/

INT32 SI_PLXC33_ParseCoff
(
	struct PLXDevice *pPLXDev, char *filename, UINT32 dspMemCount,
	UINT32 *passiveCoffImage, UINT32 *passiveCoffImageCount,
	UINT32 *activeCoffImage, UINT32 *activeCoffImageCount,
	UINT32 *wordsBeforeCoff
)
{
	INT32 error;
	UINT32 cnt, cnt1, passiveCoffImageCnt, activeCoffImageCnt;
	UINT32 *ptr;

	passiveCoffImageCnt = 0x400;

	// for coff parsing
	UINT32
		dspAddr[kCOFFFile_MaxNumberOfSections], 
		size[kCOFFFile_MaxNumberOfSections], 
		*data[kCOFFFile_MaxNumberOfSections], 
		loadableSections;
	char *sectionNames[kCOFFFile_MaxNumberOfSections];
	char *buffer;

	// Following 2 variables are created just in case future use need them.
	UINT32 *wordsAfterCoff, *wordsAfterEnd;
	wordsAfterCoff = 0;
	wordsAfterEnd = 0;

	// 2. Add extra params that loads before COFF
	ptr = wordsBeforeCoff;
	if (ptr != NULL)
	{
		if(passiveCoffImageCnt + ptr[0] >= dspMemCount)
			return e_Err_BufferTooSmall;

		for (cnt = 0; cnt < ptr[0]; cnt++)
			passiveCoffImage[passiveCoffImageCnt++] = ptr[1+cnt];	//data
	}

	// 3. parse the actual DSP coff file
	buffer = (char *)malloc(kCOFFFile_MaxBufferSize);
	if (buffer == NULL)
		return e_Err_WindowsMemError;

	// parse coff file to determine what should be loaded.
	error = 
		ParseCoff
		(
			filename, buffer, sectionNames,
			dspAddr, size, data, &loadableSections
		);
	if ( error != e_Err_NoError)
	{
		free(buffer);
		return error;
	}

	// parse all sections from COFF file.
	activeCoffImageCnt = 0;
	for (cnt = 0; cnt < loadableSections; cnt++)
	{	
		if ( dspAddr[cnt] == (kPLXC33_CoffLoadOffset) )
		{
			//	Inside of first 1k is copied blindly.
			for (cnt1 = 0; cnt1 < size[cnt]; cnt1++)
				passiveCoffImage[ cnt1 ] = data[cnt][cnt1];
		}
		else
		{
			if (strncmp(sectionNames[cnt], "a_", 2) != 0)
			{
				//	Outside of first 1k of DSP code must be loaded with header.
				if(passiveCoffImageCnt + (4 + size[cnt]) >= dspMemCount)
				{
					free(buffer);
					return e_Err_BufferTooSmall;
				}
				passiveCoffImage[passiveCoffImageCnt++] = 0;			// type
				passiveCoffImage[passiveCoffImageCnt++] = size[cnt];	// count
				passiveCoffImage[passiveCoffImageCnt++] = 0;			// src
				passiveCoffImage[passiveCoffImageCnt++] = dspAddr[cnt];	// dst
				for (cnt1=0; cnt1<size[cnt]; cnt1++)
					passiveCoffImage[passiveCoffImageCnt++] = data[cnt][cnt1];
			}
			else
			{
				activeCoffImage[activeCoffImageCnt++] = 0;			// type
				activeCoffImage[activeCoffImageCnt++] = size[cnt];	// count
				activeCoffImage[activeCoffImageCnt++] = 0;			// src
				activeCoffImage[activeCoffImageCnt++] = dspAddr[cnt];	// dst
				for (cnt1=0; cnt1<size[cnt]; cnt1++)
					activeCoffImage[activeCoffImageCnt++] = data[cnt][cnt1];
			}
		}
	}
	free(buffer);

	// 4. Add extra params that occurs after COFF
	ptr = wordsAfterCoff;
	if (ptr != NULL)
	{
		if(passiveCoffImageCnt + ptr[0] >= dspMemCount)
			return e_Err_BufferTooSmall;

		for (cnt=0; cnt<ptr[0]; cnt++)
			passiveCoffImage[passiveCoffImageCnt++] = ptr[1+cnt];	//data
	}

	// 5. write end marker
	if(passiveCoffImageCnt + 2 >= dspMemCount)
		return e_Err_BufferTooSmall;

	passiveCoffImage[passiveCoffImageCnt++] = 0xffffffff;
	passiveCoffImage[passiveCoffImageCnt++] = 0xffffffff;

	activeCoffImage[activeCoffImageCnt++] = 0xffffffff;
	activeCoffImage[activeCoffImageCnt++] = 0xffffffff;

	// 6. write stuff after end marker
	ptr = wordsAfterEnd;
	if (ptr != NULL)
	{
		if(passiveCoffImageCnt + ptr[0] >= dspMemCount)
			return e_Err_BufferTooSmall;

		for (cnt=0; cnt<ptr[0]; cnt++)
			passiveCoffImage[passiveCoffImageCnt++] = ptr[1+cnt];	//data
	}

	// 7. this many words are written to DSP
	*passiveCoffImageCount = passiveCoffImageCnt;
	*activeCoffImageCount = activeCoffImageCnt;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_CoffLoad
//
//	Description:
//		Loads coff file. Does not check.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		char *filename	: File to access.
//		UINT32 coffOptions: DSP Active or DSP Inactive.
//		UINT32 *wordsBeforeCoff: stuff that loads before actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterCoff: stuff that loads after actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterEnd: stuff that loads after end marker
//			see comments for SI_PLXC6711_ParseCoff

INT32 SI_PLXC33_CoffWrite
(
	struct PLXDevice *pPLXDev, char *filename, UINT32 coffOptions, 
	UINT32 *coffImage, UINT32 coffImageCount
)
{
	INT32 error;
	UINT32	cnt, type, size, source, destination, *data;

	// make sure the driver is in block mode, in case it's in point mode
	//	by accident.
	error = 
		SI_PLX_DriverConfig_BlockPoint
		(
			pPLXDev, kFlagDriverConfig_Block
		);
	if (error != e_Err_NoError)
		return error;

	//	DSP Active. Load using HOSTPOLL.
	if (coffOptions == e_Coffload_DSPActive)
	{
		// type & size are read separately, because they could contain
		// 0xffffffff which is the End Of File Mark, if this mark is found,
		// no more data must be read.
		cnt = 0;
		type = coffImage[cnt++];
		size = coffImage[cnt++];

		while((type != 0xffffffff) && (size != 0xffffffff))
		{
			// if no End Of File Mark, then read source and destination
			source = coffImage[cnt++];
			destination = coffImage[cnt++];
			data = &coffImage[cnt];
			cnt = cnt + size;

			error = 
				SI_PLXC33_WriteHostpoll
				(
					pPLXDev, size, destination, data
				);
			if(error != e_Err_NoError)
			{
				return error;
			}

			// read next type & size
			type = coffImage[cnt++];
			size = coffImage[cnt++];
		}

		return e_Err_NoError;
	}
	
	// DSP inactive loading.
	if (coffOptions == e_Coffload_DSPInactive)
	{
		error = 
			SI_PLXC33_ResetBoard
			(
				pPLXDev, 
				e_Reset_Assert
			);
		if ( error != e_Err_NoError)
		{
			return error;
		}

		// load all sections from COFF file, as a single write.
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev,
				0,
				coffImageCount,
				kPLXC33_CoffLoadOffset,
				coffImage
			);
		if( error != e_Err_NoError )
		{
			return error;
		}

		// write reset vector to CoffFileOffset so when reset is removed it
		// starts our si_c_int00 routine, parsing cofffile to correct locations
		cnt = kPLXC33_CoffLoadOffset;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev,
				0,
				1,
				kPLXC33_ResetVectorAddr,
				&cnt
			);
		if( error != e_Err_NoError )
		{
			return error;
		}
		return e_Err_NoError;
	}		
	return e_Err_UnknownCommand;

}


///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_CoffVerify
//
//	Description:
//		Checks the coff file with what's in DSP memory. Does not start the DSP.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		char *filename	: File to access.
//		UINT32 coffOptions: Target or hostpoll
//		UINT32 *extrasBeforeCoff: stuff that gets loaded before actual coff
//		UINT32 *extrasAfterCoff: stuff that gets loaded after actual coff
//			format for extras is type, count, src, dst, data0, ..., dataN-1
//		UINT32 *wordsBeforeCoff: stuff that loads before actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterCoff: stuff that loads after actual coff
//			see comments for SI_PLXC6711_ParseCoff
//		UINT32 *wordsAfterEnd: stuff that loads after end marker
//			see comments for SI_PLXC6711_ParseCoff

INT32 SI_PLXC33_CoffVerify
(
	struct PLXDevice *pPLXDev, char *filename, UINT32 coffOptions, 
	UINT32 *coffImage, UINT32 coffImageCount
)
{
	INT32 error;
	UINT32	*verify, cnt, cnt1, type, size, source, destination, *data;

	// make sure the driver is in block mode, in case it's in point mode
	//	by accident.
	error = 
		SI_PLX_DriverConfig_BlockPoint
		(
			pPLXDev, kFlagDriverConfig_Block
		);
	if (error != e_Err_NoError)
		return error;

	// allocate verify mem only after parsing coff to reduce memory load.
	verify = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
	if (verify == NULL)
	{
		return e_Err_WindowsMemError;
	}

	//	DSP Active. Load using HOSTPOLL.
	if (coffOptions == e_Coffload_DSPActive)
	{
		// type & size are read separately, because they could contain
		// 0xffffffff which is the End Of File Mark, if this mark is found,
		// no more data must be read.
		cnt = 0;
		type = coffImage[cnt++];
		size = coffImage[cnt++];

		while((type != 0xffffffff) && (size != 0xffffffff))
		{
			// if no Enf Of File Mark, then read source and destination
			source = coffImage[cnt++];
			destination = coffImage[cnt++];
			data = &coffImage[cnt];
			cnt = cnt + size;

			error = 
				SI_PLXC33_ReadHostpoll
				(
					pPLXDev, size, destination, verify
				);
			if(error != e_Err_NoError)
			{
				free(verify);
				return error;
			}

			for (cnt1 = 0; cnt1 < size; cnt1++)
			{
				if ( data[cnt1] != verify[cnt1] )
				{
					free(verify);
					return e_Err_CompareError;
				}
			}

			// read next type & size
			type = coffImage[cnt++];
			size = coffImage[cnt++];
		}

		free(verify);
		return e_Err_NoError;
	}

		// DSP inactive loading.
	if (coffOptions == e_Coffload_DSPInactive)
	{
		// 4. read from boot ram
		error = 
			SI_PLXC33_ReadTarget
			(
				pPLXDev,
				0,
				coffImageCount,
				kPLXC33_CoffLoadOffset,
				verify
			);
		if( error != e_Err_NoError )
		{
			free(verify);
			return error;
		}
			
		// 5. compare
		for (cnt = 0; cnt < coffImageCount; cnt++)
		{
			if ( coffImage[cnt] != verify[cnt] )
			{
				free(verify);
				return e_Err_CompareError;
			}
		}

		error = 
			SI_PLXC33_ReadTarget
			(
				pPLXDev,
				0,
				1,
				kPLXC33_ResetVectorAddr,
				verify
			);
		if( error != e_Err_NoError )
		{
			free(verify);
			return error;
		}

		// compare
		if ( verify[0] != kPLXC33_CoffLoadOffset )
		{
			free(verify);
			return e_Err_CompareError;
		}

		free(verify);
		return e_Err_NoError;
	}
	
	return e_Err_UnknownCommand;
}


///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ConvertDSPAddress
//
//	Description:
//		Converts input DSP address to PLX address for target and BM DMA.
//		C33 addresses as seen by PLX. All these are DWORD addresses.
//		These are write only. When written with 1, they toggle for 2 ECLK.
//		The Interrupt DSP address is obtained as follows -->
//			FF0001 can be decoded to -->	
//		******************************************************
//		* B23 | B22 | B21 | B20 | B19 | B18 | ......... | B0 *
//		******************************************************
//			1111 1111 0000 0000 0000 0001
//
//		Now on the plx side the upper 2 bits are page bits and
//		bits 21 to 19 are ignored 
//		******************************************************
//		* B20 | B19 | B18 | ............................| B0 *
//		******************************************************
//		so pg = 11 = 3
//		so we are left with bits 21 -> 0
//			11 1111 0000 0000 0000 0001 
//		After bits 21 -> 19 are removed,we have
//			111 0000 0000 0000 0001 
//
//		Appending the page bits
//			1 1111 0000 0000 0000 0001 
//		which is Hex
//			0x1F0001
//		translating from DWORD boundary to BYTE boundary(left shift by 2)
//			1 1111 0000 0000 0000 000100 
//		rearraging the bits
//			0111 1100 0000 0000 0000 0100 
//		which is Hex 
//			0x7C0004
//		Which is the Address used by PLX PassThrough Write to cause DSP 
//		Interrupt. The above Process also applies to CSR for DSP Reset
//		(0xFF0000)
//
//	Parameters:
//		UINT32 dspAddr	: DSP address from application.
//		UINT32 *plxAddr	: PLX address to use in Target and BM DMA transfers.

INT32 SI_PLXC33_ConvertDSPAddress
(
	UINT32 dspAddr,
	UINT32 *plxAddr
)
{
	UINT32 page;

	// DSP page
	page = (dspAddr & 0xC00000) >> 3;
	*plxAddr = (page + (dspAddr & 0x07ffff)) << 2;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadTarget
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		region			: PLX space
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_ReadTarget
(
	struct PLXDevice *pPLXDev, 
	UINT32 region, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{

	INT32 error;
	UINT32 plxAddr;

	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;

	error = SI_PLX_ReadTarget( pPLXDev, region, count, plxAddr, hostAddr );
	if ( error != e_Err_NoError)
		return error;

			
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteTarget
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		region			: PLX space
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_WriteTarget
(
	struct PLXDevice *pPLXDev, 
	UINT32 region, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxAddr;

	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;

	error = SI_PLX_WriteTarget( pPLXDev, 0, count, plxAddr, hostAddr );
	if ( error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteBlockDMA
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_WriteBlockDMA
(
	struct PLXDevice *pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxAddr;
	UINT32 timeout;

	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;

	// set timer (for long transfers)
	timeout = count / 1000 + 1000;	// 1000 word / sec + 1 sec

	error = SI_PLX_SetTimeout( pPLXDev, timeout );
	if (error != e_Err_NoError)
		return error;
	
	error = SI_PLX_WriteBlockDMA( pPLXDev, count, plxAddr, hostAddr );
	if ( error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadBlockDMA
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (DWORD offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_ReadBlockDMA
(
	struct PLXDevice *pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxAddr;
	UINT32 timeout;

	// convert addr to DSP
	error = SI_PLXC33_ConvertDSPAddress( dspAddr, &plxAddr );
	if ( error != e_Err_NoError)
		return error;
	
	// set timer (for long transfers)
	timeout = count / 1000 + 1000;	// 1000 word / sec + 1 sec

	error = SI_PLX_SetTimeout( pPLXDev, timeout );
	if (error != e_Err_NoError)
		return error;

	error = 
		SI_PLX_ReadBlockDMA
		(
			pPLXDev, 
			count, 
			plxAddr,
			hostAddr
		);
	if ( error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_TranslateCommModeIndex
//
//	Description:
//		Converts commModeIndex (consecutive numbers) to actual commMode bits
//		that the DSP can use. Also returns the "name" of the corresponding
//		comm mode.
//		
//	Parameters:
//		UINT32 commModeIndex 	: CommMode index (consecutive number)
//		UINT32 *commMode		: Return value. Converted from comm Index.
//		UINT32 maxStringCount	: Max number of bytes allocated in String.
//		char *commName			: Pointer to string where the name will reside.

INT32 SI_PLXC33_TranslateCommModeIndex
(
	UINT32 commModeIndex, UINT32 *commMode, 
	UINT32 maxStringCount, char *commName

)
{
	UINT32 commMode1;
	char *stringVal = NULL;
	
	switch (commModeIndex)
	{

	case e_CommModes_PLXC33_Target:
		commMode1 =	-1;
		stringVal = "Host: Target.	DSP: N/A";
		break;

	case e_CommModes_PLXC33_BMDMA:
		commMode1 =	-1;
		stringVal = "Host: BMDMA.	DSP: N/A";
		break;

	case e_CommModes_PLXC33_Addon_Async_IO_Int:
		commMode1 =	kCommModeMask_HostAddonInit 
			|	kCommModeMask_InterruptHost;
		stringVal = "Add-on: DSP Async IO - Host Int.";
		break;

	case e_CommModes_PLXC33_Addon_Async_DMA_Int:
		commMode1 =	kCommModeMask_HostAddonInit 
			|	kCommModeMask_InterruptHost
			|	kCommModeMask_DMA;
		stringVal = "Add-on: DSP Async DMA - Host Int.";
		break;

	case e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg:
		commMode1 =	kCommModeMask_PointTransfer
			|	kCommModeMask_Sync;
		stringVal = "Target: DSP Sync IO CommReg";
		break;

	default:
		return e_Err_UnknownCommand;
	}

	*commMode = commMode1;

	// copy string name if there's room
	if ( (strlen (stringVal) + 1) < maxStringCount)
	{
		while ( (*stringVal) != 0 )
			*commName++ = *stringVal++;

		*commName++ = 0;
	}

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadHostpoll
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_ReadHostpoll
(
	struct PLXDevice *pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	return 
		SI_PLXC33_ReadDSPActive
		(
			pPLXDev, 
			e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg, 
			count, dspAddr, hostAddr
		);
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteHostpoll
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Pointer to host memory where the data resides.

INT32 SI_PLXC33_WriteHostpoll
(
	struct PLXDevice *pPLXDev, 
	UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	return 
		SI_PLXC33_WriteDSPActive
		(
			pPLXDev, 
			e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg, 
			count, dspAddr, hostAddr
		);
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadDSPActive
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commModeIndex: Transfer mode as modeIndex (see enum).

INT32 SI_PLXC33_ReadDSPActive
(
	struct PLXDevice *pPLXDev, 
	UINT32 commModeIndex, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 srcAddr, dstAddr, plxAddr, commMode;

	if (count == 0)
		return e_Err_NoError;

	error = 
		SI_PLXC33_TranslateCommModeIndex
		(
			commModeIndex, &commMode, 0, NULL
		);
	if ( error != e_Err_NoError )
		return error;

	// DSP write, host read.
	commMode &= ~kCommModeMask_DSPRead;

	// reading from DSP means srcAddr is the addr we want to read.
	srcAddr = dspAddr;

	// Depending on commMode, determine what should be done.
	if (commMode & kCommModeMask_PointTransfer)
	{
		// hostpoll: sync IO
		dstAddr = kPLXC33_CommReg_Data;
		plxAddr = kPLXC33_CommReg_Data;
	}
	else
	{
		// block transfer. only for addoninit
		dstAddr = kPLXC33_CommReg_HostAddr;
		plxAddr = 0;
	}

	error = 
		SI_PLXC33_DSPActiveComm1
		(
			pPLXDev, 
			commMode, count, srcAddr, dstAddr, 
			plxAddr, hostAddr
		);
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteDSPActive
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 dspAddr	: DSP address (target address) in UINT32 offset.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commModeIndex: Transfer mode as modeIndex (see enum).

INT32 SI_PLXC33_WriteDSPActive
(
	struct PLXDevice *pPLXDev, 
	UINT32 commModeIndex, UINT32 count, UINT32 dspAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 srcAddr, dstAddr, plxAddr, commMode;

	if (count == 0)
		return e_Err_NoError;

	error = 
		SI_PLXC33_TranslateCommModeIndex
		(
			commModeIndex, &commMode, 0, NULL
		);
	if ( error != e_Err_NoError )
		return error;

	// DSP read, host write
	commMode |= kCommModeMask_DSPRead;

	// write to DSP means dstAddr is the addr we want to write.
	dstAddr = dspAddr;

	// Depending on commMode, determine what should be done.
	if (commMode & kCommModeMask_PointTransfer)
	{
		// hostpoll: sync IO
		srcAddr = kPLXC33_CommReg_Data;
		plxAddr = kPLXC33_CommReg_Data;
	}
	else
	{
		// block transfer. only for addoninit
		srcAddr = kPLXC33_CommReg_HostAddr;
		plxAddr = 0;
	}

	error = 
		SI_PLXC33_DSPActiveComm1
		(
			pPLXDev, 
			commMode, count, srcAddr, dstAddr, 
			plxAddr, hostAddr
		);
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm1
//
//	Description:
//		Root level of DSPActiveComm function: DSP communications when DSP is 
//		active. Note that modes must be carefully selected or the board will 
//		lock.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 commMode	: Transfer mode. Use OR of the following:
//			kCommModeMask_InterruptHost 
//			kCommModeMask_PointTransfer
//			kCommModeMask_DSPRead 
//			kCommModeMask_Sync 
//			kCommModeMask_DMA
//			kCommModeMask_HostAddonInit 
//			kCommModeMask_HostBMDMA
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 srcAddr	: DSP source address.
//		UINT32 dstAddr	: DSP destination address.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.


INT32 SI_PLXC33_DSPActiveComm1
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 srcAddr, UINT32 dstAddr, 
	UINT32 plxAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 plxCommAddr, interruptCSRValue, wordCount, cnt;

	// get addr to commreg as PLX will see them.
	error = 
		SI_PLXC33_ConvertDSPAddress
		(
			kPLXC33_CommReg,
			&plxCommAddr
		);
	if ( error != e_Err_NoError)
		return error;

	if (commMode & kCommModeMask_PointTransfer)
	{
		// point transfer cannot be the following
		if 
		(
			commMode & 
			(
					kCommModeMask_HostAddonInit
				|	kCommModeMask_DMA
				|	kCommModeMask_HostBMDMA
			)
		)
			return e_Err_UnknownCommand;

		// point must be sync
		if ( !(commMode & kCommModeMask_Sync) )
			return e_Err_UnknownCommand;

		// Point transfer is always assumed to be to/from CommReg.

		// setup the DSP
		error = 
			SI_PLX_SetupAddonInitParams
			(
				pPLXDev, 
				plxCommAddr, 
				commMode, count, srcAddr, dstAddr
			);
		if ( error != e_Err_NoError )
			return error;

		// now actually generate interrupt
		interruptCSRValue = kPLXC33_IntDSPValue;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_IntDSPAddr, &interruptCSRValue
			);
		if ( error != e_Err_NoError )
			return error;

		// do actual transfer.
		error = 
			SI_PLXC33_DSPActiveComm2_Point
			(
				pPLXDev, 
				commMode, count, plxAddr, hostAddr
			);

		return error;
	}

	// block transfer is always to addoninit memory.
	if ( !(commMode & kCommModeMask_HostAddonInit) )
		return e_Err_UnknownCommand;

	// block transfer can never be synchronized to flag.
	//	Using flag could cause lockup due to race condition.
	if (commMode & kCommModeMask_Sync)
		return e_Err_UnknownCommand;

	// block mode must be done through host interrupt. otherwise, 
	//	it can lock.
	if ( !(commMode & kCommModeMask_InterruptHost) )
		return e_Err_UnknownCommand;
	 
	if (pPLXDev->addonInitBufferSize == 0)
		return e_Err_CountTooBig;

	if ( count > pPLXDev->addonInitBufferSize )
	{
		// chop up the transfer into multiple since it won't fit in one.
		wordCount = count;
		for (cnt=0; cnt<(wordCount / pPLXDev->addonInitBufferSize); cnt++)
		{
			// setup the DSP
			error = 
				SI_PLX_SetupAddonInitParams
				(
					pPLXDev, 
					plxCommAddr, 
					commMode, pPLXDev->addonInitBufferSize, srcAddr, dstAddr
				);
			if ( error != e_Err_NoError )
				return error;

			// do actual transfer. 
			error = 
				SI_PLXC33_DSPActiveComm2_Block
				(
					pPLXDev, 
					commMode, pPLXDev->addonInitBufferSize, plxAddr, hostAddr
				);
			if (error != e_Err_NoError)
				return error;

			count -= pPLXDev->addonInitBufferSize;
			hostAddr += pPLXDev->addonInitBufferSize;
			if (commMode & kCommModeMask_DSPRead)
			{
				// DSP read, host write, change dst addr
				dstAddr += pPLXDev->addonInitBufferSize;
			}
			else
			{
				// DSP write, host read, change srcaddr
				srcAddr += pPLXDev->addonInitBufferSize;
			}
		}	// end for many buffers
	}	// end if larger than pPLXDev->addonInitBufferSize

	// make sure this check is done after large count addoninit.
	if (count == 0)
		return e_Err_NoError;

	// setup the DSP
	error = 
		SI_PLX_SetupAddonInitParams
		(
			pPLXDev, 
			plxCommAddr, 
			commMode, count, srcAddr, dstAddr
		);
	if ( error != e_Err_NoError )
		return error;

	// do normal block transfer. 
	error = 
		SI_PLXC33_DSPActiveComm2_Block
		(
			pPLXDev, 
			commMode, count, plxAddr, hostAddr
		);
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm2_Point
//
//	Description:
//		Level 0 of DSPActiveComm function.
//		Transfers using point of DSP, such as FIFO or commreg.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commMode	: Transfer mode.

INT32 SI_PLXC33_DSPActiveComm2_Point
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 cnt, flag;

	// now read/write data using flag.

	//	DSP is using IO and synchronizing with flag. 

	// Make sure DSP got the parameters ok.
	// wait for flag
	error = 
		SI_PLXC33_DSPActiveComm4_WaitFlag
		(
			pPLXDev, 
			kCommModeFlag_ValidCommand, kCommModeTimeoutValue
		);
	if ( error != e_Err_NoError )
		return error;

	// read from DSP: set flag so that read data is ready.
	if ( !(commMode & kCommModeMask_DSPRead) )
	{
		// flag DSP that host is ready for first word to read
		flag = kCommModeFlag_HostReady;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_CommReg_Flag, &flag
			);
		if ( error != e_Err_NoError )
			return error;
	}	// end if for reading first word from DSP.

	// do the rest of the transfers.
	for (cnt=0; cnt<count; cnt++)
	{
		if ( !(commMode & kCommModeMask_DSPRead) && (cnt == count-1) )
		{
			// if this is last DSPread (host write), 
			//	don't wait for flag.
		}
		else
		{
			// wait for flag
			error = 
				SI_PLXC33_DSPActiveComm4_WaitFlag
				(
					pPLXDev, 
					kCommModeFlag_ValidCommand, kCommModeTimeoutValue
				);
			if ( error != e_Err_NoError )
				return error;
		}

		// if host addr is not NULL, then do transfer. only the point transfer 
		//	is valid and block transfer must have hostAddr set to NULL.
		if (hostAddr != NULL)
		{
			// Transfer one word just before setting the flag. 
			//	This should be FIFO or commreg_data location. count must be 1.
			error = 
				SI_PLXC33_DSPActiveComm4_ReadWrite
				(
					pPLXDev, 
					commMode, 1, plxAddr, &hostAddr[cnt]
				);
			if ( error != e_Err_NoError )
				return error;
		}

		// if this is read from DSP, and it's the last transfer, don't set flag
		if ( !(commMode & kCommModeMask_DSPRead) && (cnt == count-1) )
			break;

		// flag DSP that host is ready for another word
		flag = kCommModeFlag_HostReady;
		error = 
			SI_PLXC33_WriteTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_CommReg_Flag, &flag
			);
		if ( error != e_Err_NoError )
			return error;

	}	// end for

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm2_Block
//
//	Description:
//		Level 0 of DSPActiveComm function.
//		Transfers using block of DSP.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commMode	: Transfer mode.

INT32 SI_PLXC33_DSPActiveComm2_Block
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
)
{
	UINT32 plxIntAddr, intValue;
	INT32 error;

	error = SI_PLXC33_ConvertDSPAddress(kPLXC33_IntDSPAddr, &plxIntAddr);
	if (error != e_Err_NoError)
		return error;

	intValue = kPLXC33_IntDSPValue;

	// this always assumes addon-init, host interrupt, no sync per word.
	if (commMode & kCommModeMask_DSPRead)	// host writes
	{
		return
			SI_PLX_WriteAddonInit
			(
				pPLXDev, 
				plxIntAddr, intValue, 
				count, hostAddr
			);
	}
	else
	{
		return
			SI_PLX_ReadAddonInit
			(
				pPLXDev, 
				plxIntAddr, intValue, 
				count, hostAddr
			);
	}

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm4_ReadWrite
//
//	Description:
//		Level 2 of DSPActiveComm function.
//		Depending on commMode, read/write using target/BMDMA.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to transfer.
//		UINT32 plxAddr	: PLX addr to use during transfer.
//		UINT32 *hostAddr: Host addr where the data resides.
//		UINT32 commMode	: Mode of read/write, target/BMDMA.

INT32 SI_PLXC33_DSPActiveComm4_ReadWrite
(
	struct PLXDevice *pPLXDev, 
	UINT32 commMode, UINT32 count, UINT32 plxAddr, UINT32 *hostAddr
)
{
	INT32 error;

	// read from DSP. This should be FIFO location.
	if (commMode & kCommModeMask_DSPRead)	// host writes
	{
		if (commMode & kCommModeMask_HostBMDMA)
			error = 
				SI_PLXC33_WriteBlockDMA
				(
					pPLXDev, 
					count, plxAddr, hostAddr
				);
		else
			error = 
				SI_PLXC33_WriteTarget
				(
					pPLXDev, 
					0, count, plxAddr, hostAddr
				);
	}
	else								// host reads
	{
		if (commMode & kCommModeMask_HostBMDMA)
			error = 
				SI_PLXC33_ReadBlockDMA
				(
					pPLXDev, 
					count, plxAddr, hostAddr
				);
		else
			error = 
				SI_PLXC33_ReadTarget
				(
					pPLXDev, 
					0, count, plxAddr, hostAddr
				);
	}	// end if DSP read

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_DSPActiveComm4_WaitFlag
//
//	Description:
//		Level 4 of DSPActiveComm function.
//		Waits for flag value until timout.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 flagValue: Flag to wait for.
//		UINT32 timeoutMSec: Timout value in msec.

INT32 SI_PLXC33_DSPActiveComm4_WaitFlag
(
	struct PLXDevice *pPLXDev, 
	UINT32 flagValue, UINT32 timeoutMSec
)
{
	INT32 error;
	UINT32 cnt, flag, sleepTime;

	sleepTime = 100;	// time to sleep between successive checks.

	// wait for flag
	for (cnt=0; 1; cnt++)
	{
		error = 
			SI_PLXC33_ReadTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_CommReg_Flag, &flag
			);
		if ( error != e_Err_NoError )
			return error;

		if ( flag == flagValue )
			break;	// valid flag received

		SI_PLX_Sleep(sleepTime);					// give some time

		if ( ( cnt * sleepTime ) > timeoutMSec )
			return e_Err_AddonInitTimeoutError;	// error.
	}

	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	configure memory
//	

INT32 SI_PLXC33_CoffLoad_GetMemoryParams
(
	struct PLXDevice *pPLXDev, UINT32 *memoryParams
)
{
	UINT32 testValue, cnt, bank0Size, bank1Size, temp;
	INT32 error;
	UINT32 const Bank0Start = 0;
	UINT32 const Bank1Start = 0x400000;

	// Addresses below are possible end locations of each bank depending on 
	// size (option) of memory available.The first 2 memory addresses are for
	// bank 0, the other 2 for bank 1, if more options are included
	// in the future, the number of addresses will be 'bankOptions' X #banks,
	// these addresses must be arranged from lowest to highest for each bank.
	
	UINT32 memorySizes[2] = {0x20000, 0x80000};		// 128K, 512K
	UINT32 bankOptions = 2;
	
	UINT32 rareValue = 0x55550000;		// use a rare value to test memory
	UINT32 dummyRareValue = 0xAAAAAAAA;	// another value to avoid latched lines
	

	bank0Size = bank1Size = 0;

	for(cnt = 0; cnt < bankOptions; cnt++)
	{
		temp = rareValue + cnt;
		error = SI_PLXC33_WriteTarget
				(
					pPLXDev, 
					0, 1, Bank0Start + memorySizes[cnt] - 1, &temp
				);
		if(error)
			return error;

		if(cnt == 0)					// check if bank 0 exists
		{
			// This write is to avoid reading exactly what we just wrote
			error = SI_PLXC33_WriteTarget
					(
						pPLXDev, 
						0, 1, Bank0Start + memorySizes[0] - 2, &dummyRareValue
					);
			if(error)
				return error;

			error = SI_PLXC33_ReadTarget
				(
					pPLXDev, 
					0, 1, Bank0Start + memorySizes[0] - 1, &testValue
				);
			if(error)
				return error;

			// set minimum size for this bank, if exists
			if(testValue == rareValue + cnt)
				bank0Size = memorySizes[cnt];
		}

		if(cnt == 0)			// check if bank 1 exists
		{
			// This write is to avoid reading exactly what we just wrote
			error = SI_PLXC33_WriteTarget
					(
						pPLXDev, 
						0, 1, Bank1Start + memorySizes[0] - 2, &dummyRareValue
					);
			if(error)
				return error;

			error = SI_PLXC33_ReadTarget
				(
					pPLXDev, 
					0, 1, Bank1Start + memorySizes[0] - 1, &testValue
				);
			if(error)
				return error;

			// set minimum size for this bank, if exists
			if(testValue == rareValue + cnt)
				bank1Size = memorySizes[cnt];
		}
	}

	// if memorybank0 exists
	if(bank0Size)
	{
		// check for size of Bank0
		for(cnt = 0; cnt < bankOptions - 1; cnt++)
		{
			error = SI_PLXC33_ReadTarget
				(
					pPLXDev, 
					0, 1, Bank0Start + memorySizes[cnt] - 1, &testValue
				);
			if(error)
				return error;

			if(testValue != rareValue + cnt)
				break;
		}
		bank0Size = memorySizes[cnt];

		// check for size of Bank1, if exists
		if(bank1Size)
		{
			for(cnt = 0; cnt < bankOptions - 1; cnt++)
			{
				error = SI_PLXC33_ReadTarget
					(
						pPLXDev, 
						0, 1, Bank1Start + memorySizes[cnt] - 1, &testValue
					);
				if(error)
					return error;

				if(testValue != rareValue + cnt)
					break;
			}
			bank1Size = memorySizes[cnt];
		}
		memoryParams[0] = bank0Size;
		memoryParams[1] = bank1Size;

	return e_Err_NoError;
	}
	else
	{
		memoryParams[0] = 0;
		memoryParams[1] = 0;

// Note: Change following error code to e_Err_UnknownMemoryConfig
		return e_Err_BufferTooSmall;			// No Memory Present
	}
}


///////////////////////////////////////////////////////////////////////////////
//	UINT32 SI_PLXC33_RecvMessage
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.

INT32 SI_PLXC33_RecvMessage
(
	struct PLXDevice *pPLXDev, INT32 waitMilliSecs, UINT32 *msg
)
{
	UINT32 flag;
	INT32 error;

	if (waitMilliSecs != 0) {
		error = SI_PLX_WaitMessageFromDSP(pPLXDev, waitMilliSecs);
		if ( error != e_Err_NoError )
			return error;
	}

	//check for message
	error = 
		SI_PLXC33_ReadTarget
		(
			pPLXDev, 
			0, 1, kPLXC33_DSPMessageFlag, &flag
		);
	if ( error != e_Err_NoError )
		return error;

	//if no message present return error
	if(flag != kPLXC33_RecvMsgFlag)
		return e_Err_RecvMsg_NoMsg;

	//read the message
	error = 
		SI_PLXC33_ReadTarget
		(
			pPLXDev, 
			0, 1, kPLXC33_DSPMessage, msg
		);
	if ( error != e_Err_NoError )
		return error;

	//clear message flag
	flag = 0;
	error = 
		SI_PLXC33_WriteTarget
		(
			pPLXDev, 
			0, 1, kPLXC33_DSPMessageFlag, &flag
		);
	if ( error != e_Err_NoError )
		return error;

	
	return e_Err_NoError;	// interrupt was detected.

}


///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_SendMessage
//
//	Description:
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.

INT32 SI_PLXC33_SendMessage
(
	struct PLXDevice *pPLXDev, INT32 waitMilliSecs, UINT32 msg
)
{
	UINT32 data, cnt = 0;;
	INT32 error;

	cnt = waitMilliSecs;

	while(TRUE)
	{
		//check if message already pending
		error = 
			SI_PLXC33_ReadTarget
			(
				pPLXDev, 
				0, 1, kPLXC33_HostMessageFlag, &data
			);
		if ( error != e_Err_NoError )
			return error;

		if ( data == 0 )
			break;	// valid flag received

		if (cnt >= 0)
		{
			if (cnt-- == 0)
			{
				return e_Err_SendMsg_Pending;	// error.
			}
		}

		SI_PLX_Sleep(1);					// give some time
	}

	//write the message
	error = 
		SI_PLXC33_WriteTarget
		(
			pPLXDev, 
			0, 1, kPLXC33_HostMessage, &msg
		);
	if ( error != e_Err_NoError )
		return error;

	data = kPLXC33_SendMsgFlag;
	//write message flag
	error = 
		SI_PLXC33_WriteTarget
		(
			pPLXDev, 
			0, 1, kPLXC33_HostMessageFlag, &data
		);
	if ( error != e_Err_NoError )
		return error;

	//Generate DSP interrupt
	error = SI_PLXC33_CommInterrupt(pPLXDev);
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;		
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_CommInterrupt
//
//	Description:
//		Interrupt to start DSP C6711 communications
//
//	Parameters:
//		None
//
//	Notes:
//		03-04-03 Created. ILL.

INT32 SI_PLXC33_CommInterrupt(struct PLXDevice *pPLXDev)
{
	UINT32 interruptCSRValue;

	interruptCSRValue = kPLXC33_IntDSPValue;
	return 
		SI_PLXC33_WriteTarget
		(
			pPLXDev, 
			0, 1, kPLXC33_IntDSPAddr, &interruptCSRValue
		);
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_WriteTargetBootMem
//
//	Description:
//		
//		PLX uses byte addressing. COFF files use byte addressing, but COFF
//		images use DWord addressing.
//
//		For e_Coffload_DSPInactive: 
//			Only lower 8 bits are used for BOOT RAM.
//			Therefore, each 32 bit write result in 8 bits of data transfer.
//			For example, if the COFF file tries to write 0x12345678 
//				to addr 0x0010,
//				COFF ADDR	PLX ADDR	PLX DATA	EA		ED
//				0x0010		0x0040		0x00000078	0x0010	0x78
//				0x0011		0x0044		0x00000056	0x0011	0x56
//				0x0012		0x0048		0x00000034	0x0012	0x34
//				0x0013		0x004C		0x00000012	0x0013	0x12
//		Note that the addon addr increments 16 bytes for each word.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 addonAddr: Addon address (byte offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.
//
/*
INT32 SI_PLXC33_WriteTargetBootMem
(
	struct PLXDevice *pPLXDev, 
	UINT32 count, UINT32 addonAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 cnt, cnt1, tempAddr, tempData[4];
	tempAddr = addonAddr;

	// write actual data
	for (cnt=0; cnt<count; cnt++)
	{
		// breakup the word into 4 bytes
		for (cnt1=0; cnt1<4; cnt1++)
		{
			tempData[cnt1] = (hostAddr[cnt]) >> (cnt1 * 8);
			tempData[cnt1] &= 0xff;
		}
		// write those words using one driver call (faster).
		error = 
			SI_PLX_WriteTarget
			(
				pPLXDev, 
				0, 4, tempAddr, tempData
			);
		if( error != e_Err_NoError )
			return error;

		// increment addr by 4 for each byte, 16 per word.
		tempAddr += 4 * 4;	
	}

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	INT32 SI_PLXC33_ReadTargetBootMem
//
//	Description:
//		PLX uses byte addressing. COFF files use byte addressing. This reads
//		collection of 8bit numbers to 32bit RAM
//		This is only visible in this file.
//
//		For e_Coffload_DSPInactive: 
//			Only lower 8 bits are used for BOOT RAM.
//			Therefore, each 32 bit write result in 8 bits of data transfer.
//			For example, if the COFF file tries to write 0x12345678 
//				to addr 0x0010,
//				COFF ADDR	PLX ADDR	PLX DATA	EA		ED
//				0x0010		0x0040		0x00000078	0x0010	0x78
//				0x0011		0x0044		0x00000056	0x0011	0x56
//				0x0012		0x0048		0x00000034	0x0012	0x34
//				0x0013		0x004C		0x00000012	0x0013	0x12
//		Note that the addon addr increments 16 bytes for each word.
//
//	Parameters:
//		struct PLXDevice *pPLXDev	: Handle to the device driver.
//		UINT32 count	: Number of UINT32 to access.
//		UINT32 addonAddr: Addon address (byte offset).
//		UINT32 *hostAddr: Pointer to host memory where the data resides.
//
INT32 SI_PLXC33_ReadTargetBootMem
(
	struct PLXDevice *pPLXDev, 
	UINT32 count, UINT32 addonAddr, UINT32 *hostAddr
)
{
	INT32 error;
	UINT32 cnt, cnt1, tempAddr, tempData[4];

	tempAddr = addonAddr;

	for (cnt=0; cnt<count; cnt++)
	{
		// read 4 words with one driver call (faster)
		error = 
			SI_PLX_ReadTarget
			(
				pPLXDev, 
				0, 4, tempAddr, tempData
			);					
		if( error != e_Err_NoError )
			return error;

		// increment addr by 4 for each byte, 16 per word.
		tempAddr += 4 * 4;

		// pack 4 bytes into one word.
		hostAddr[cnt] = 0;
		for (cnt1=0; cnt1<4; cnt1++)
		{
			tempData[cnt1] &= 0xff;
			hostAddr[cnt] |= ( tempData[cnt1] << ( cnt1 * 8 ) );

		}	// end do this 4 times for each byte of word
	}


	return e_Err_NoError;
}
*/
///////////////////////////////////////////////////////////////////////////////

