///////////////////////////////////////////////////////////////////////////////
//	PLX_FPGALoad.cpp
//
//	Description:
//		Loads FPGA using PLX signals, USERo, EESK, EEDO, USERi.
//
//	Revision History:
//		2002-04-12: mik
//			Created.
//		2002-04-22: mik
//			Added support for all drivers (95, NT, WDM)
//		2002-07-09: mik
//			Added FPGA type parameter and scanning for parameter.
//		2002-09-25: mik
//			Code is now platform neutral (only using ANSI code).
//		2003-08-26: mik
//			Modularize all funcs to support cascaded FPGA.
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../common/sihw_common_plx.h"

#define kFPGA_ParamStart	2
#define kFPGA_ParamsPerFPGA	2
#define kFPGA_Param_Type	0
#define kFPGA_Param_Size	1

///////////////////////////////////////////////////////////////////////////////
//	main
//
//	Description: Main program for the FPGA Loader.
//
//	Parameters:
//		int argc:		Count of arguments from Command Line, arguments include
//						the command itself, in this case: PLX_FPGALoad.exe.
//		char *argv[]:	Array of arguments, for this function, there are 3,
//						the first one is always PLX_FPGALoad.exe, followed by
//						slot number of the board to be Bit Loaded, then
//						complete path for the Bit Description File.
//

int main(int argc, char *argv[])
{
	int error;
	UINT32 boardID;
	char *descriptionFilePath;
	struct PLXDevice hPLXDev, *pPLXDev = &hPLXDev;


	// Display the Banner
	printf("\nSI FPGA Load, version 3\n");
	printf("Sheldon Instruments, Inc. 2007\n");
	printf("Supporting the PLX 90xx PCI chip\n\n");

	error =
		ParseCommandLine
		(
			argc, argv, &boardID, &descriptionFilePath
		);
	if (error != 0)
	{
		DoError("");
		return 1;
	}

	// ok. open driver.
	error = 
		SI_PLX_OpenDriver(boardID, pPLXDev);
	if (error != e_Err_NoError)
	{
		printf("\n\nCould not open the driver.\nLoading FPGA");

		DoError("");
		return 1;
	}

	error = LoadBoardFPGAs(boardID, pPLXDev, descriptionFilePath);
	if (error != e_Err_NoError)
	{
		SI_PLX_CloseDriver(pPLXDev);
		DoError("");
		return 1;
	}

	// When done, close the handle to the driver
	if( SI_PLX_CloseDriver(pPLXDev) != e_Err_NoError )
	{
		printf("The driver handle could not be unlinked properly\nLoading FPGA");
		return 1;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
//	DoError
//
//	Description: Print out error in case there were an error and board
//				 would not work.
//
//	Parameters:
//		char *errorMsg:	Message describing the error.
//

void DoError(char *errorMsg)
{
	printf("%s", errorMsg);
	printf("\nSome FPGA were not loaded. The board might not function.\n");
	printf("Press enter to continue.\n");
	getchar();

	return;
}




