///////////////////////////////////////////////////////////////////////////////
//	coffParse.cpp
//
//	Sheldon Instruments, Inc.
//
//	Description:
//		Parses any and all COFF files and returns values that can be used to
//		download to the DSP. From external function, call ParseCoff.Then, it
//		returns the following.
//			- DSP address where the code should be loaded
//			- Number of words to load
//			- Data to load
//			- Number of sections to be loaded
//
//	Functions:
//		COFF_DebugWriteToFile		
//		ParseCoff01Sections
//		ParseCoff2Sections
//		ParseCoff
//
//	Revision History:
//		2001-11-06: mik
//			Created
//		2002-03-05: mik
//			Fixed the loading of dummy section.
//		2002-03-06: mik
//			Removed UINT8. basetsd.h from MS screws things up!
//		2002-04-17: mik
//			Shuffled #define paths to reflect directory changes.
//		2002-06-25: mik
//			Fixed problem with ParseCoff fread size comparison.
//		2004-03-05: Ley
//			For compiling with C6711 BIOS (provided by TI), it adds
//				a section named .vers with STYP_COPY set. It triggers
//				a portion of program in ParseCoff2Sections which
//				causes DSP to crash. A condition was added to avoid
//				.vers to trigger this portion of program.
//		2006-04-06: Whipple
//			Added coffOPtions to ParseCoff() and ParseCoff2Sections()
//		2006-07-03: Whipple
//			Combined active and passive coffload
//			section names are now parsed for an active label
//		2006-09-01: Whipple
//			Fixed flag check for executable
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include "coffparse.h"
#include "../common_ddk/sierrors.h"
#include "sicodes.h"

////////////////////////////////////////////////////////////////////////////////
//	int COFF_DebugWriteToFile
//
//	Description:
//		Parses a coff file and writes the result into easily readable text 
//		format. This is useful for debugging and determining what type of COFF
//		file is being used.
//
//	Parameters:
//		coffFile: string to the name of the COFF file to parse.
//		textFile: string to the name of the text file to generate.
//
//	Return values:
//		error codes
//
int COFF_DebugWriteToFile
(
	char *coffFile, char *textFile
)
{
	char buffer[kCOFFImage_MaxBufferSize];
	char *sectionNames[kCOFFFile_MaxNumberOfSections];
	UINT32 dspAddr[kCOFFFile_MaxNumberOfSections];
	UINT32 size[kCOFFFile_MaxNumberOfSections]; 
	UINT32 *data[kCOFFFile_MaxNumberOfSections];
	UINT32 loadableSections;
	UINT32 cnt;
	UINT32 cnt1;

	int error;
	FILE *fp;

	error = ParseCoff(coffFile, buffer, sectionNames,
			dspAddr, size, data, &loadableSections);
	if (error != e_Err_NoError)
		return error;

	fp = fopen(textFile, "wt");
	if (fp == NULL)
		return e_Err_FileNotFound;

	for (cnt=0; cnt<loadableSections; cnt++)
	{
		fprintf(fp, "DSP ADDR = 0x%x\n", dspAddr[cnt]);
		fprintf(fp, "Section Size = %d\n", size[cnt]);

		for (cnt1=0; cnt1<size[cnt]; cnt1++)
			fprintf(fp, "	0x%x", data[cnt][cnt1]);

		fprintf(fp, "\n\n");
	}
	fclose (fp);
	return e_Err_NoError;
}

////////////////////////////////////////////////////////////////////////////////
//	int ParseCoff01Sections
//
//	Description:
//		Parses COFF sections after the header and optional header are
//		determined. After parsing, all relevant data are stored in the array.
//
//		Applicable to COFF0 and COFF1
//		0-7 Character 
//			8-character section name, padded with nulls
//		8-11 Long integer 
//			Sections physical address
//		12-15 Long integer 
//			Sections virtual address
//		16-19 Long integer 
//			Section size in words
//		20-23 Long integer 
//			File pointer to raw data
//		24-27 Long integer 
//			File pointer to relocation entries
//		28-31 Long integer 
//			File pointer to line number entries
//		32-33 Unsigned short integer 
//			Number of relocation entries
//		34-35 Unsigned short integer 
//			Number of line number entries
//		36-37 Unsigned short integer 
//			Flags (see Table A-6 in spru035)
//		38 Character 
//			Reserved
//		39 Unsigned character 
//			Memory page number
//
//	Parameters:
//		buffer: input
//			kCOFFImage_MaxBufferSize bytes of data that contains the raw COFF.
//		ptr: input
//			Current byte-offset pointer within the buffer. It must point to
//			the first sections header. 0 means first byte of buffer.
//		sectionsCount: input
//			Total number of sections present within this COFF file.
//		dspAddr: output
//			Array to store the starting DSP address for each section.
//		size: output
//			Array to store the size for each section.
//		data: output
//			Array to store the pointer to actual data.
//		loadableSections: output
//			Number of loadable sections present within the COFF.
//			Note that this value is returned from this function.
//
//	Return values:
//		error codes
//
//	NOTE: 
//		buffer and all output must be allocated at the same level. For 
//		example, if buffer is deallocated, data will be pointing to invalid 
//		memory location. Therefore, always allocate/deallocate them at the 
//		same time.
//
int ParseCoff01Sections
(
	char *buffer, UINT32 ptr, UINT16 sectionsCount, char *sectionNames[],
	UINT32 dspAddr[], UINT32 size[], UINT32 *data[], UINT32 *loadableSections
)
{
	char	*sectName;
	INT32	addrPhysical;
	INT32	addrVirtual;
	INT32	sectSize;
	INT32	ptr2Data;
	INT32	ptr2Reloc;
	INT32	ptr2LineNumbers;
	UINT16	relocCount;
	UINT16	lineNumberCount;
	UINT16	flags;
	INT32	cnt, idx;

	idx = 0; // index for loadable sections
	for (cnt=0; cnt<(INT32)sectionsCount; cnt++)
	{
		sectName		= (char *)(buffer + ptr);	ptr += 8;
		addrPhysical	=*(INT32 *)(buffer + ptr);	ptr += 4;
		addrVirtual		=*(INT32 *)(buffer + ptr);	ptr += 4;
		sectSize		=*(INT32 *)(buffer + ptr);	ptr += 4;
		ptr2Data		=*(INT32 *)(buffer + ptr);	ptr += 4;
		ptr2Reloc		=*(INT32 *)(buffer + ptr);	ptr += 4;
		ptr2LineNumbers	=*(INT32 *)(buffer + ptr);	ptr += 4;
		relocCount		=*(UINT16 *)(buffer + ptr);	ptr += 2;
		lineNumberCount	=*(UINT16 *)(buffer + ptr);	ptr += 2;
		flags			=*(UINT16 *)(buffer + ptr);	ptr += 2;
		ptr += 1;	// reserved
		ptr += 1;	// mem page number

		// if size is 0, nothing to do
		if ( sectSize == 0)
			continue;

		// if the ptr2Data is 0, it should not be loadable
		if ( ptr2Data == 0)
			continue;

		// if the type is not loadable, then skip section
		if 
		(
				( flags & STYP_DSECT )	// dummy
			|	( flags & STYP_NOLOAD )	// no load
			|	( flags & STYP_BSS )	// BSS
		)
			continue;
		if 
		(
			!(
				(flags == STYP_REG) || (flags & STYP_PAD)  || 
				(flags & STYP_TEXT) || (flags & STYP_DATA)
			)
		)
			continue;

		// what remains are loadable sections. record them.
		if 	( flags & STYP_COPY )
		{
			// for ".vers" see notes on header of file
			// STYP_COPY is messy
			if (strcmp(sectName, ".vers") != 0)
			{
				while( 1 )
				{
					sectSize = *(UINT32 *)(buffer + ptr2Data);
					ptr2Data += sizeof (UINT32);
					addrVirtual = *(UINT32 *)(buffer + ptr2Data);
					ptr2Data += sizeof (UINT32);

					// if size is 0, nothing to do
					if ( sectSize == 0)
						break;

					sectionNames[idx] = sectName;
					dspAddr[idx] = addrVirtual;
					size[idx] = sectSize;
					data[idx] = (UINT32 *)(buffer + ptr2Data);
					idx++;

					ptr2Data += sectSize * sizeof (UINT32);
				}
			}
		}
		else
		{
			sectionNames[idx] = sectName;
			dspAddr[idx] = addrVirtual;
			size[idx] = sectSize;
			data[idx] = (UINT32 *)(buffer + ptr2Data);			
			idx ++;
		}
	}

	*loadableSections = idx;
	return e_Err_NoError;
}

////////////////////////////////////////////////////////////////////////////////
//	int ParseCoff2Sections
//
//	Description:
//		Parses COFF sections after the header and optional header are
//		determined. After parsing, all relevant data are stored in the array.
//
//		Applicable to COFF2
//		0-7 Character 
//			8-character section name, padded with nulls
//		8-11 Long integer 
//			Sections physical address
//		12-15 Long integer 
//			Sections virtual address
//		16-19 Long integer 
//			Section size in words
//		20-23 Long integer 
//			File pointer to raw data
//		24-27 Long integer 
//			File pointer to relocation entries
//		28-31 Long integer 
//			File pointer to line number entries
//		32-35 Unsigned long
//			Number of relocation entries
//		36-39 Unsigned long integer 
//			Number of line number entries
//		40-43 Unsigned long integer 
//			Flags (see Table A-6 in spru035)
//		44-45 Character 
//			Reserved
//		46-47 Unsigned character 
//			Memory page number
//
//	Parameters:
//		buffer: input
//			kCOFFImage_MaxBufferSize bytes of data that contains the raw COFF.
//		ptr: input
//			Current byte-offset pointer within the buffer. It must point to
//			the first sections header. 0 means first byte of buffer.
//		sectionsCount: input
//			Total number of sections present within this COFF file.
//		dspAddr: output
//			Array to store the starting DSP address for each section.
//		size: output
//			Array to store the size for each section.
//		data: output
//			Array to store the pointer to actual data.
//		loadableSections: output
//			Number of loadable sections present within the COFF.
//			Note that this value is returned from this function.
//		coffOptions: input
//			type of coffload passive or passive+active
//
//	Return values:
//		error codes
//
//	NOTE: 
//		buffer and all output must be allocated at the same level. For 
//		example, if buffer is deallocated, data will be pointing to invalid 
//		memory location. Therefore, always allocate/deallocate them at the 
//		same time.
//
int ParseCoff2Sections
(
	char *buffer, UINT32 ptr, UINT16 sectionsCount, char *sectionNames[],
	UINT32 dspAddr[], UINT32 size[], UINT32 *data[], UINT32 *loadableSections
)
{
	char	*sectName;
	INT32	addrPhysical;
	UINT32	addrVirtual;			//Whipple INT32 -> UINT32
	INT32	sectSize;
	INT32	ptr2Data;
	INT32	ptr2Reloc;
	INT32	ptr2LineNumbers;
	UINT32	relocCount;
	UINT32	lineNumberCount;
	UINT32	flags;
	INT32	cnt, idx;

	idx = 0; // index for loadable sections
	for (cnt=0; cnt<(INT32)sectionsCount; cnt++)
	{
		sectName		= (char *)(buffer + ptr);	ptr += 8;
		addrPhysical	=*(INT32 *)(buffer + ptr);	ptr += 4;
		addrVirtual		=*(INT32 *)(buffer + ptr);	ptr += 4;
		sectSize		=*(INT32 *)(buffer + ptr);	ptr += 4;
		ptr2Data		=*(INT32 *)(buffer + ptr);	ptr += 4;
		ptr2Reloc		=*(INT32 *)(buffer + ptr);	ptr += 4;
		ptr2LineNumbers	=*(INT32 *)(buffer + ptr);	ptr += 4;
		relocCount		=*(UINT32 *)(buffer + ptr);	ptr += 4;
		lineNumberCount	=*(UINT32 *)(buffer + ptr);	ptr += 4;
		flags			=*(UINT32 *)(buffer + ptr);	ptr += 4;
		ptr += 2;	// reserved
		ptr += 2;	// mem page number

		// if size is 0, nothing to do
		if ( sectSize == 0)
			continue;

		// if the ptr2Data is 0, it should not be loadable
		if ( ptr2Data == 0)
			continue;

		// if the type is not loadable, then skip section
		if 
		(
				( flags & STYP_DSECT )	// dummy
			|	( flags & STYP_NOLOAD )	// no load
			|	( flags & STYP_BSS )	// BSS
		)
			continue;
		if 
		(
			!(
				(flags == STYP_REG) || (flags & STYP_PAD)  || 
				(flags & STYP_TEXT) || (flags & STYP_DATA)
			)
		)
			continue;

		// what remains are loadable sections. record them.
		if 	( flags & STYP_COPY )
		{
			// for ".vers" see notes on header of file
			// STYP_COPY is messy
			if (strcmp(sectName, ".vers") != 0)
			{
				while( 1 )
				{
					sectSize = *(UINT32 *)(buffer + ptr2Data);
					ptr2Data += sizeof (UINT32);
					addrVirtual = *(UINT32 *)(buffer + ptr2Data);
					ptr2Data += sizeof (UINT32);

					// if size is 0, nothing to do
					if ( sectSize == 0)
						break;

					sectionNames[idx] = sectName;
					dspAddr[idx] = addrVirtual;
					size[idx] = sectSize;
					data[idx] = (UINT32 *)(buffer + ptr2Data);
					idx++;

					ptr2Data += sectSize * sizeof (UINT32);
				}
			}
		}
		else
		{
				// what remains are loadable sections. record them.
				sectionNames[idx] = sectName;
				dspAddr[idx] = addrVirtual;
				size[idx] = sectSize;
				data[idx] = (UINT32 *)(buffer + ptr2Data);
				idx++;
		}
	}

	*loadableSections = idx;
	return e_Err_NoError;
}

////////////////////////////////////////////////////////////////////////////////
//	int ParseCoff
//
//	Description:
//		Parses raw COFF file and returns the result in array.
//		Following is the description of the file header. Offsets are in bytes.
//		0-1	Unsigned short integer 
//			COFF 0: Contains magic number (093h) to indicate the file can be 
//				executed in a TMS320C3x/C4x system.
//			COFF 1 and 2: Contains COFF version number, either 0c1h (COFF1) 
//				or 0c2h (COFF 2).
//		2-3	Unsigned short integer 
//			Number of section headers
//		4-7 Long integer 
//			Time and date stamp; indicates when the file was created
//		8-11 Long integer
//			File pointer; contains the symbol tables starting address
//		12-15 Long integer 
//			Number of entries in the symbol table
//		16-17 Unsigned short integer 
//			Number of bytes in the optional header. This field is either 0 or 
//			28; if it is 0, there is no op-tional file header.
//		18-19 Unsigned short integer 
//			Flags (see Table A-2 in spru035)
//		20-21 Unsigned short integer 
//			Included for COFF 1 and 2 only. Contains magic number (093h) to 
//			indicate the file can be executed in a TMS320C3x/C4x system.
//
//	Parameters:
//		filename: input
//			COFF file name string.
//		buffer: output
//			kCOFFImage_MaxBufferSize bytes of array to hold COFF file.
//		dspAddr: output
//			Array to store the starting DSP address for each section.
//		size: output
//			Array to store the size (in words) for each section.
//		data: output
//			Array to store the pointer to actual data (derived from buffer).
//		loadableSections: output
//			Number of loadable sections present within the COFF.
//			Note that this value is returned from this function.
//		coffOptions: input
//			Identifies coff parse method
//
//	Return values:
//		error codes
//
//	NOTE: 
//		buffer and all output must be allocated at the same level. For 
//		example, if buffer is deallocated, data will be pointing to invalid 
//		memory location. Therefore, always allocate/deallocate them at the 
//		same time.
//		If there is COFF3, COFF4, COFF314, etc., each can be added by using 
//		the case statement below and adding functions such as 
//		ParseCoff314Sections.
//
int ParseCoff
(
	char *filename, char *buffer, char *sectionNames[], UINT32 dspAddr[],
	UINT32 size[], UINT32 *data[], UINT32 *loadableSections
)
{
	UINT32	ptr = 0;
	UINT16	coffType;				
	UINT16	sectionsCount;
	UINT32	timeDateStamp;
	UINT32	symbolTablePointer;
	UINT32	symbolEntriesCount;
	UINT16	optionalHeaderBytesCount;
	UINT16	flags;
	UINT16	coff12MagicNumber;
	UINT32 count;
	// fill up the buffer. this is now using ANSI
	FILE *fp;

	fp = fopen(filename, "rb");
	if ( fp == NULL )
		return e_Err_FileNotFound;

	count = fread(buffer, sizeof(char), kCOFFFile_MaxBufferSize / sizeof(char), fp);
	fclose(fp);
	
	if(count == kCOFFFile_MaxBufferSize / sizeof(char))
		return e_Err_BufferTooSmall;

	// get the params from the buffer
	coffType					= *(UINT16 *)(buffer + ptr);	ptr += 2;
	sectionsCount				= *(UINT16 *)(buffer + ptr);	ptr += 2;
	timeDateStamp				= *(UINT32 *)(buffer + ptr);	ptr += 4;// can ignore
	symbolTablePointer			= *(UINT32 *)(buffer + ptr);	ptr += 4;// can ignore
	symbolEntriesCount			= *(UINT32 *)(buffer + ptr);	ptr += 4;// can ignore
	optionalHeaderBytesCount	= *(UINT16 *)(buffer + ptr);	ptr += 2;
	flags						= *(UINT16 *)(buffer + ptr);	ptr += 2;

	// start error checking
	if ( ! ( flags & 2 ) )		// is not executable
		return e_Err_CoffTypeError;

	if ( sectionsCount > kCOFFFile_MaxNumberOfSections )
		return e_Err_CoffSectionsError;

	// parse sections depending on COFF file type
	switch(coffType)
	{
	case 0xC1:	// COFF1
		//coff12MagicNumber = 0x93 means it is C3x/C4x executable
		coff12MagicNumber		= *(UINT16 *)(buffer + ptr);	ptr += 2;
		// COFF1 is same as COFF0 except for magic number, so NO BREAK!!!
	case 0x93:	// COFF0
		// skip optional header
		ptr += optionalHeaderBytesCount;	

		return	ParseCoff01Sections
				(
					buffer, ptr, sectionsCount,  sectionNames,
					dspAddr, size, data, loadableSections
				);
		break;

	case 0xC2:	// COFF2
		//coff12MagicNumber = 0x93 means it is C3x/C4x executable
		coff12MagicNumber		= *(UINT16 *)(buffer + ptr);	ptr += 2;

		// skip optional header
		ptr += optionalHeaderBytesCount;	

		return	ParseCoff2Sections
				(
					buffer, ptr, sectionsCount, sectionNames,
					dspAddr, size, data, loadableSections
				);
		break;
	
	default:
		return e_Err_CoffTypeError;
	}
}

/*
////////////////////////////////////////////////////////////////////////////////
//	INT32 FileSize
//
//	Description:
//		Funtion to find the number of bytes in a file and compare 
//		to a bufffer size.
//
//	Returns:
//		0 = buffer size greater file size
//		1 = file size greater than buffer size
//

INT32 FileSize(FILE *fp, UINT32 bufsize)
{
	UINT32 count = 0, total = 0;
	UINT32 buffer[100];

	while( !feof( fp ) )
	{
		count = 0;
		// Attempt to read in 100 bytes:
		count = fread(buffer, sizeof(char), 100, fp);
		
		total += count;
		if( ferror( fp ) )      
			break;
	}

	fseek(fp, 0, SEEK_SET);

	if(bufsize > total)
		return 0;
	else
		return 1;

}
*/
