////////////////////////////////////////////////////////////////////////////////
//	CPLXC33.cpp
//
//	Sheldon Instruments, Inc.
//
//	Abstract:
//		Class Functions for supporting the PLX Board for PQ\VQ.
//
//	Functions:
//		CPLXC33::ClassLoad	
//		CPLXC33::ClassUnload
//		CPLXC33::CoffLoad
//		CPLXC33::ResetBoard
//		CPLXC33::Configure
//		CPLXC33::ReadFromBoard
//		CPLXC33::WriteToBoard
//		CPLXC33::ReadGeneric
//		CPLXC33::WriteGeneric	
//		CPLXC33::CoffLoad_ConfigMemory
//		CPLXC33::CoffQuListfromFile
//		CPLXC33::CloseQuListFileandLaunch
//
//	Revision History:
//
//		1999-01-12: Robert Updike
//			modified
//		2001-03-09: mik
//			modified
//		2001-03-20: mik
//			major surgery fixes various problems with NT and 98. 
//			Works except for 64 channel scope with DSPDMA writes under NT
//		2001-05-15: mik
//			Overhauled Add-on-Initiated to have host-poll-mode
//			and COFF loading
//		2001-08-24: mik
//			Removed unnecessary #includes.
//		2001-09-04: mik
//			Added reset features (Assert, Deassert, Toggle)
//		2001-11-08: mik
//			Changed Get/Set NV Bytes to return word intead of byte.
//		2001-11-15: mik
//			Cleaned up comment section
//		2002-01-17: mik
//			Added ReadPCI_NVWord, WritePCI_NVWord.
//			Added ReadPCI_OpReg, WritePCI_OpReg.
//		2002-01-25: mik
//			Added memory type to Configure
//		2002-06-28: mik
//			Changed rest of functions to use API.
//		2002-07-01: mik
//			Changed all instances of SIHW_ to SI_
//			Changed C33PLXC9054 to CPLXC33.
//		2002-07-08: mik
//			Using OverlappedEventsCreate to create all events.
//			Changed cgpOverlappedMisc to Read/Write as appropriate.
//			Fixed all comments. Return values are now SI error code.
//		2002-09-16: mik
//			Moved all pOverlapped and callback stuff into API.
//		2002-09-16: mik
//			Minor clean up.
//		2002-10-07: mik
//			Minor clean up.
//		2004-04-28: Ley
//			Relocated / Expanded QLists from:	Before		Now
//			#define	ckDSP_BaseAddrFel0			0x4000		0x1000;
//			#define ckDSP_BaseAddrPqf0			0x4800		0x2000;
//			#define ckDSP_BaseAddrBHead0		0x5000		0x3000;
//			#define ckDSP_BaseAddrBlock0		0x5800		0x4000;
//		2004-09-07: Ley
//			Added before Coff values. These are VQ list Start / End marks.
//			Removed from DSP VQLoop Initialization and placed as before Coff
//			items to make C6711 CoffLoad from Flash.
//		2004-09-17: Ley
//			Modified so it configures memory size by getting memory available,
//			instead of using the Manual Control Setting.
//		2006-10-16: Ley
//			Expanded GetBoardInfo so now it also outputs
//			DSP Bus Speed and Boundary Factor.
//		2006-12-12: AV
//			Added Coffloading from quListFile functionality for C33.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef vqhost_EXPORTS
#include "../../siqux/32_BIT_DLL/Rev1/CQuXboard.h"	
// addresses for PQ\VQ on DSP. These are DSP addresses
const UINT32 kPLXC33_BaseAddrFel = 0x1000;
const UINT32 kPLXC33_BaseAddrPqf = 0x2000;
const UINT32 kPLXC33_BaseAddrBHead = 0x3000;
const UINT32 kPLXC33_BaseAddrBlock = 0x4000;
#else
#include "cboard.h"
const UINT32 cgDSP_BaseAddrFel = 0;	// When QuX is not included.
#endif
#include <stdlib.h>

#include "../siddk_plx/apps/sihw_plx/Rev1/c33/common/sicommon_plxc33.h"

UINT32 const kPLXC33_ParseableDataStart = 0x1000;

extern struct PLXC33 board;
struct PLXDevice cgSIDev;

///////////////////////////////////////////////////////////////////////////////
//
//	Routine Description:
//
//		Constructor for PLX9054 Board.
//
//	Arguments:
//
//		param[0] = board to open (0, 1, 2, 3, etc)
//

INT32 ClassLoad( UINT32 *params )
{
	INT32 error = 0;
	
	board.pBoardInfo = &board.boardInfo;
//	cgpSIDev = new PLXDevice;
	board.cgpSIDev = &cgSIDev;
	

	board.boardInfo.boardID = params[0];

	// get handle to driver.
	error = SI_PLXC33_OpenDriver( board.pBoardInfo, board.cgpSIDev );
	if (error != e_Err_NoError)
	{
//		delete board.cgpSIDev;
//		board.cgpSIDev = NULL;
		return error;
	}

	board.cgReadModeBelowLimit	= e_CommModes_Invalid;
	board.cgReadModeAboveLimit	= e_CommModes_Invalid;
	board.cgWriteModeBelowLimit	= e_CommModes_Invalid;
	board.cgWriteModeAboveLimit	= e_CommModes_Invalid;

#ifdef vqhost_EXPORTS
	// addresses for PQ\VQ on DSP. These are DSP addresses
	cgDSP_BaseAddrFel	= kPLXC33_BaseAddrFel;
	cgDSP_BaseAddrPqf	= kPLXC33_BaseAddrPqf;
	cgDSP_BaseAddrBHead	= kPLXC33_BaseAddrBHead;
	cgDSP_BaseAddrBlock	= kPLXC33_BaseAddrBlock;
	cgDSP_MaxBlockAddr	= 0x1FFFF;	// default to 128k

	// Initialize quListFile name.
	board.quListFile = NULL;
#endif

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//
//	Routine Description:
//	
//	    Call this function before exiting the DLL. 
//		This is needed since the class destructor doesn't work.
//	
//	Arguments:
//	
//	    none
//	
INT32 ClassUnload()
{
	INT32 error;

#ifdef vqhost_EXPORTS
	if(board.quListFile != NULL)
	{
		free(board.quListFile);
		// Initialize quListFile name.
		board.quListFile = NULL;
	}
#endif

	if(board.cgpSIDev != NULL)
	{
		error = SI_PLX_CloseDriver(board.cgpSIDev);
//		delete board.cgpSIDev;
//		board.cgpSIDev = NULL;
	}

	return error;
}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    Loads the COFF on the DSP. DSP does not yet run until reset is toggled.
//	
//	Arguments:
//	
//		filename[]	: COFF file name.
//	    coffOptions	: See enum for e_Coffload. 
//						Could be DSP active or DSP inactive.
//						Now is bypassed and forced to have Passive,
//						then Active, if any. ILL 07/27/06
//		delay		: Not used.
//	

INT32 CoffLoad(
			char filename[], UINT32 coffOptions, UINT32 delay, UINT32 launch, UINT32 *coffImageSize)
{
	INT32 error;
	UINT32 	*beforeCoff, beforeCoffCount, 
			// only for file based coff load
			dspMemCount = kCOFFImage_MaxBufferSize / sizeof(UINT32),
			*passiveCoffImage, passiveCoffImageCount,
			*activeCoffImage, activeCoffImageCount, cnt;
		
	UINT32 const BB_END_MARK = 1000;
	UINT32 memoryParams[2];			// Use array to match C6711 SPD data format

	beforeCoff = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
	if (beforeCoff == NULL)
		return e_Err_WindowsMemError;

	error = SI_PLXC33_CoffLoad_GetMemoryParams(board.cgpSIDev, memoryParams);
	if(error) return error;

	board.pBoardInfo->cgDSP_Bank0 = memoryParams[0];
	board.pBoardInfo->cgDSP_Bank1 = memoryParams[1];

	error = CoffLoad_ConfigMemory(memoryParams);
	if(error) return error;

	// parameters to write before COFF file is written
	beforeCoffCount = 1;	// reserve first for count

	// if something must be added before Coff file, add it here and
	// increment beforCoffCount++ per each item, Follow following example

	// 09/07/04 Following is added to make C33 compatible
	// with Flash CoffLoad of C6711.
	// these are DSP Start / VQ list Stop/End marks.
	// If CoffLoading from Flash, these are the default values at
	// the begining of the VQ list. If there's VQ list, these values
	// will be overwritten by the list. Either there's PQ and / or VQ
	// list, the first Data '0' (DSP Start mark) will be overwritten
	// to indicate the DSP to start right after Reset is Toggled.
	// Overwritting these values are done by DSP when parsing
	// PQ/VQ list from Flash.
	beforeCoff[beforeCoffCount++] = 3;					//type
	beforeCoff[beforeCoffCount++] = 3;					//count
	beforeCoff[beforeCoffCount++] = 0;					//src
	beforeCoff[beforeCoffCount++] = cgDSP_BaseAddrFel;	//dst
	beforeCoff[beforeCoffCount++] = 0x0;				//DSP Start mark
	beforeCoff[beforeCoffCount++] = 0x0;				//data
	beforeCoff[beforeCoffCount++] = BB_END_MARK;		//VQ list End mark

	// actual number of words is one less than total (first is count)
	beforeCoffCount --;
	beforeCoff[0] = beforeCoffCount;

	//Allocate memory for passiveCoffImage and activeCoffImage,
	//and initialize all them to zeros.
	passiveCoffImage = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
	if (passiveCoffImage == NULL)
	{
		free(beforeCoff);
		return e_Err_WindowsMemError;
	}

	activeCoffImage = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
	if (activeCoffImage == NULL)
	{
		free(beforeCoff);
		free(passiveCoffImage);
		return e_Err_WindowsMemError;
	}

	for(cnt = 0; cnt < dspMemCount; cnt++)
	{
		activeCoffImage[cnt] = 0;
		passiveCoffImage[cnt] = 0;
	}

	//Create QuList file, then CoffLoad until Terminate.
	if(board.coffType == 1)
	{
		if(board.quListfp == NULL)
			{
				if((board.quListfp = fopen(board.quListFile, "wb+")) == NULL)
					{
						free(activeCoffImage);
						free(passiveCoffImage);
						free(beforeCoff);
						return e_Err_CreateFile;
					}
			}

			board.savingQuListFile = 1;

		// now parse the coff file
		error = SI_PLXC33_ParseCoff
				(
					board.cgpSIDev, filename, dspMemCount,
					passiveCoffImage, &passiveCoffImageCount,
					activeCoffImage, &activeCoffImageCount,
					beforeCoff
				);
		if(error != e_Err_NoError)
		{
			free(activeCoffImage);
			free(passiveCoffImage);
			free(beforeCoff);			
			return error;
		}

		*coffImageSize = board.listSize = board.quListFileSize = (passiveCoffImageCount + activeCoffImageCount);
	
		// Write passive part to file
		passiveCoffImageCount -= 2;		//retrieve end mark since we still add PQ/VQ list.

		if((fwrite(passiveCoffImage, sizeof(UINT32),
			passiveCoffImageCount, board.quListfp)) !=
			passiveCoffImageCount)
		{
			free(activeCoffImage);
			free(passiveCoffImage);
			free(beforeCoff);
			return e_Err_FileWriteError;
		}

		// Attach active part to file
		activeCoffImageCount -= 2;		//retrieve end mark since we still add PQ/VQ list.

		if((fwrite(activeCoffImage, sizeof(UINT32),
				activeCoffImageCount, board.quListfp)) !=
				activeCoffImageCount)
			{
				free(activeCoffImage);
				free(passiveCoffImage);
				free(beforeCoff);
				return e_Err_FileWriteError;
			}

			free(activeCoffImage);
			free(passiveCoffImage);
			free(beforeCoff);
			return e_Err_NoError;

	}
	else if(board.coffType == 2)	// Coff directly from an existing QuList file)
	{
		error = CoffQuListfromFile(board.quListFile);
		if(error)
		{
			free(activeCoffImage);
			free(passiveCoffImage);
			free(beforeCoff);
			return error;
		}

		*coffImageSize = board.quListFileSize;

		error = SI_PLXC33_ResetBoard( board.cgpSIDev, e_Reset_Toggle );
		if (error)
		{
			free(activeCoffImage);
			free(passiveCoffImage);
			free(beforeCoff);
			return error;
		}

		free(activeCoffImage);
		free(passiveCoffImage);
		free(beforeCoff);
		return e_Err_NoError;

	}
	else			//Coff and run normal
	{
		// Default mode: Normal
		// not to file, but to actual DSP mem
			
		// now parse the coff file
		error = SI_PLXC33_ParseCoff
				(
					board.cgpSIDev, filename, dspMemCount,
					passiveCoffImage, &passiveCoffImageCount,
					activeCoffImage, &activeCoffImageCount,
					beforeCoff
				);
		if(error != e_Err_NoError)
		{
			free(activeCoffImage);
			free(passiveCoffImage);
			free(beforeCoff);			
			return error;
		}

		*coffImageSize = board.listSize = (passiveCoffImageCount + activeCoffImageCount);

		// Start Writing Passive portion of Coff file to DSP
		error = SI_PLXC33_CoffWrite
				(
					board.cgpSIDev, filename, 0,
					passiveCoffImage, passiveCoffImageCount
				);
		if (error != e_Err_NoError)
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}
		
		error = SI_PLXC33_CoffVerify
				(
					board.cgpSIDev, filename, 0,
					passiveCoffImage, passiveCoffImageCount
				);
		if (error != e_Err_NoError)
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}

		// End Passive Coff Load
		if(launch)
		{
			//Toggle reset of DSP so it starts and we can active
			//coffload the rest of the Coff file.
			error = SI_PLXC33_ResetBoard
			(
				board.cgpSIDev, 
				e_Reset_Toggle	
			);
			if(error != e_Err_NoError)
			{
				free(activeCoffImage);
				free(passiveCoffImage);
				free(beforeCoff);
				return error;
			}

			// Start write Active portion of Coff file
			// Only if DSP has launched
			// Note that if no active sections are defined,
			// SI_PLXC33_CoffWrite & SI_PLXC33_CoffVerify are called,
			// but nothing is written to the DSP.

			// now actually write to DSP
			error = SI_PLXC33_CoffWrite
					(
						board.cgpSIDev, filename, 1,
						activeCoffImage, activeCoffImageCount
					);
			if (error != e_Err_NoError)
			{
				free(beforeCoff);
				free(activeCoffImage);
				free(passiveCoffImage);
				return error;
			}
		
			error = SI_PLXC33_CoffVerify
					(
						board.cgpSIDev, filename, 1,
						activeCoffImage, activeCoffImageCount
					);
			if (error != e_Err_NoError)
			{
				free(beforeCoff);
				free(activeCoffImage);
				free(passiveCoffImage);
				return error;
			}

			// End Active Coff Load
		}
	}

	free(beforeCoff);
	free(activeCoffImage);
	free(passiveCoffImage);

	return e_Err_NoError;
}

/*
INT32 CPLXC33::CoffLoad(char filename[], UINT32 coffOptions, UINT32 delay)
{
	INT32 error;

	error = SI_PLXC33_CoffLoad( cgpSIDev, filename, coffOptions );
	if (error != e_Err_NoError)
		return error;

	error = SI_PLXC33_CoffVerify( cgpSIDev, filename, coffOptions );
	if (error != e_Err_NoError)
		return error;

	return e_Err_NoError;
}
*/

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//		Resets the board, or resets and holds the board in reset.
//	
//	Arguments:
//	
//	    resetMode	: See enum e_ResetMode
//	
	
INT32 ResetBoard(UINT32 resetMode) 
{
	return SI_PLXC33_ResetBoard( board.cgpSIDev, resetMode );
}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    Configures the board with user defined parameters.
//	
//	Arguments:
//	
//		params	: Array containing parameters.
//				  See idxXXX for description on various words.
//	

INT32 Configure(UINT32 *params)
{
	UINT32 
		idxReadModeBelowLimit = 2, 
		idxReadModeAboveLimit = 3, 
		idxReadBreakLimit = 4, 

		idxWriteModeBelowLimit = 5, 
		idxWriteModeAboveLimit = 6, 
		idxWriteBreakLimit = 7; 
		
	board.cgReadModeBelowLimit = params[idxReadModeBelowLimit];
	board.cgReadModeAboveLimit = params[idxReadModeAboveLimit];
	board.cgReadBreakLimit = params[idxReadBreakLimit];

	board.cgWriteModeBelowLimit = params[idxWriteModeBelowLimit];
	board.cgWriteModeAboveLimit = params[idxWriteModeAboveLimit];
	board.cgWriteBreakLimit = params[idxWriteBreakLimit];

	// check if these are within bounds.
	if
	(
			(board.cgReadModeBelowLimit < e_CommModes_PLXC33_Target)
		||	(board.cgReadModeBelowLimit > e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg)
		||	(board.cgReadModeAboveLimit < e_CommModes_PLXC33_Target)
		||	(board.cgReadModeAboveLimit > e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg)
		||	(board.cgWriteModeBelowLimit < e_CommModes_PLXC33_Target)
		||	(board.cgWriteModeBelowLimit > e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg)
		||	(board.cgWriteModeAboveLimit < e_CommModes_PLXC33_Target)
		||	(board.cgWriteModeAboveLimit > e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg)
	)
	{		
		board.cgReadModeBelowLimit = e_CommModes_Invalid;
		board.cgReadModeAboveLimit = e_CommModes_Invalid;

		board.cgWriteModeBelowLimit = e_CommModes_Invalid;
		board.cgWriteModeAboveLimit = e_CommModes_Invalid;

		return e_Err_UnknownCommand;
	}

	//NOTE: 0x400000 is beginning of PAGE1. Therefore, for two bank memory, 
	//	Addr = 0x400000 - (page 1 size) + offsetvalue
	//	Size = 0x400000 - (page 1 size) + total memory

/*
	// memory type
	switch(params[idxMemoryType])
	{
	case 0:	// 128k at PAGE0
		cgDSP_BaseAddrFel	= ckDSP_BaseAddrFel0;
		cgDSP_BaseAddrPqf	= ckDSP_BaseAddrPqf0;
		cgDSP_BaseAddrBHead	= ckDSP_BaseAddrBHead0;
		cgDSP_BaseAddrBlock	= ckDSP_BaseAddrBlock0;
		cgDSP_MaxBlockAddr	= 0x1FFFF;
		break;

	case 1:	// 512k at PAGE0
		cgDSP_BaseAddrFel	= ckDSP_BaseAddrFel0;
		cgDSP_BaseAddrPqf	= ckDSP_BaseAddrPqf0;
		cgDSP_BaseAddrBHead	= ckDSP_BaseAddrBHead0;
		cgDSP_BaseAddrBlock	= ckDSP_BaseAddrBlock0;
		cgDSP_MaxBlockAddr	= 0x7FFFF;
		break;

	case 2: // 128k at PAGE0, 128k at PAGE1		
		cgDSP_BaseAddrFel	= 0x400000 - 0x20000 + ckDSP_BaseAddrFel0;
		cgDSP_BaseAddrPqf	= 0x400000 - 0x20000 + ckDSP_BaseAddrPqf0;
		cgDSP_BaseAddrBHead	= 0x400000 - 0x20000 + ckDSP_BaseAddrBHead0;
		cgDSP_BaseAddrBlock	= 0x400000 - 0x20000 + ckDSP_BaseAddrBlock0;
		cgDSP_MaxBlockAddr	= 0x400000 - 0x20000 + 0x3FFFF;
		break;

	case 3: // 128k at PAGE0, 512k at PAGE1
		cgDSP_BaseAddrFel	= 0x400000 - 0x20000 + ckDSP_BaseAddrFel0;
		cgDSP_BaseAddrPqf	= 0x400000 - 0x20000 + ckDSP_BaseAddrPqf0;
		cgDSP_BaseAddrBHead	= 0x400000 - 0x20000 + ckDSP_BaseAddrBHead0;
		cgDSP_BaseAddrBlock	= 0x400000 - 0x20000 + ckDSP_BaseAddrBlock0;
		cgDSP_MaxBlockAddr	= 0x400000 - 0x20000 + 0x9FFFF;
		break;

	case 4: // 512k at PAGE0, 128k at PAGE1
		cgDSP_BaseAddrFel	= 0x400000 - 0x80000 + ckDSP_BaseAddrFel0;
		cgDSP_BaseAddrPqf	= 0x400000 - 0x80000 + ckDSP_BaseAddrPqf0;
		cgDSP_BaseAddrBHead	= 0x400000 - 0x80000 + ckDSP_BaseAddrBHead0;
		cgDSP_BaseAddrBlock	= 0x400000 - 0x80000 + ckDSP_BaseAddrBlock0;
		cgDSP_MaxBlockAddr	= 0x400000 - 0x80000 + 0x9FFFF;
		break;

	case 5: // 512k at PAGE0, 512k at PAGE1
		cgDSP_BaseAddrFel	= 0x400000 - 0x80000 + ckDSP_BaseAddrFel0;
		cgDSP_BaseAddrPqf	= 0x400000 - 0x80000 + ckDSP_BaseAddrPqf0;
		cgDSP_BaseAddrBHead	= 0x400000 - 0x80000 + ckDSP_BaseAddrBHead0;
		cgDSP_BaseAddrBlock	= 0x400000 - 0x80000 + ckDSP_BaseAddrBlock0;
		cgDSP_MaxBlockAddr	= 0x400000 - 0x80000 + 0xfffff;
		break;

	default:	// 128k at PAGE0
		cgDSP_MaxBlockAddr	= 0x1FFFF;
		return e_Err_UnknownCommand;
	}
*/

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    Transfers data from DSP memory to host memory.
//	
//	Arguments:
//	
//	    count	: Number of words to be transferred.
//	    dspAddr	: Address to read from.
//		hostAddr: Address to read to.
//	

INT32 ReadFromBoard
(
	UINT32 count, 
	UINT32 dspAddr, 
	UINT32 *hostAddr
)
{
	UINT32 mode;

	if (count == 0)
		return e_Err_NoError;

	INT32 error;
	UINT32 size, addr, offset;
	UINT32 temp = 0;

	if(board.savingQuListFile == 1)
	{
		//set file pointer to beginning
		fseek(board.quListfp, kPLXC33_ParseableDataStart, SEEK_SET);
		//search file for that dsp addr and size
		while(!ferror(board.quListfp))
		{
			//goto and read the count
			fseek(board.quListfp, sizeof(UINT32), SEEK_CUR);
			if(fread(&size, sizeof(UINT32), 1, board.quListfp) != 1)
				break;

			//goto and read the dsp addr
			fseek(board.quListfp, sizeof(UINT32), SEEK_CUR);
			if(fread(&addr, sizeof(UINT32), 1, board.quListfp) != 1)
				break;

			if
			(
				((addr + size * board.boardInfo.cgDSP_DataSize) >=
				(dspAddr + count * board.boardInfo.cgDSP_DataSize)) 
				&& (dspAddr >= addr)
			)
			{	
				offset = dspAddr - addr;	

				temp = (UINT32)ftell(board.quListfp);
				if(offset > 0)
					temp += offset;
			
			}

			fseek(board.quListfp, sizeof(UINT32) * size, SEEK_CUR);
		}

		if(temp)
		{
			fseek(board.quListfp, temp, SEEK_SET);
			if((fread(hostAddr, sizeof(UINT32), count, board.quListfp)) != count)
				return e_Err_FileReadError;
			return e_Err_NoError;
		}
		else
			return e_Err_UnableToReadFile;
	}
	else
	{

		// if greater than 0 do passthrough
		if(count <= board.cgReadBreakLimit)
			mode = board.cgReadModeBelowLimit;
		else
			mode = board.cgReadModeAboveLimit;

		// comm mode when greater than limit.
		switch (mode)
		{
		// These 2 are PLX communication modes, not DSP
		case e_CommModes_PLXC33_Target:
			return 
				SI_PLXC33_ReadTarget
				(
					board.cgpSIDev, 
					0, count, dspAddr, hostAddr
				);

		case e_CommModes_PLXC33_BMDMA:
			return
				SI_PLXC33_ReadBlockDMA
				(
					board.cgpSIDev, 
					count, dspAddr, hostAddr
				);

		// The rest are DSP active modes
		default:
			return 
				SI_PLXC33_ReadDSPActive
				(
					board.cgpSIDev, 
					mode, 
					count, dspAddr, hostAddr
				);
		}

		return error;
	}	

}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    Transfers data from host memory to DSP memory.
//	
//	Arguments:
//	
//	    count	: Number of words to be transferred.
//	    dspAddr	: Address to write to.
//		hostAddr: Address to write from.
//	

INT32 WriteToBoard
(
	UINT32 count, 
	UINT32 dspAddr, 
	UINT32 *hostAddr
)
{
	UINT32 mode;
	INT32 error;
	UINT32 temp = 0;

	if (count == 0)
		return e_Err_NoError;
	
	board.listSize += count;

	if(board.savingQuListFile == 1)
	{
		fseek(board.quListfp, 0, SEEK_END);

		if((fwrite(&board.dataType, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(&count, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(&temp, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(&dspAddr, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(hostAddr, sizeof(UINT32), count, board.quListfp)) != count)
			return e_Err_FileWriteError;

		return e_Err_NoError;
	}
	else
	{
		
		// if greater than 0 do passthrough
		if(count <= board.cgWriteBreakLimit)
			mode = board.cgWriteModeBelowLimit;
		else
			mode = board.cgWriteModeAboveLimit;

		// do comm
		switch (mode)
		{
		// These 2 are PLX communication modes, not DSP
		case e_CommModes_PLXC33_Target:
			return 
				SI_PLXC33_WriteTarget
				(
					board.cgpSIDev, 
					0, count, dspAddr, hostAddr
				);

		case e_CommModes_PLXC33_BMDMA:
			return
				SI_PLXC33_WriteBlockDMA
				(
					board.cgpSIDev, 
					count, dspAddr, hostAddr
				);

		// The rest are DSP active modes
		default:
			return 
				SI_PLXC33_WriteDSPActive
				(
					board.cgpSIDev, 
					mode, 
					count, dspAddr, hostAddr
				);
		}

		return error;
	}

}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    Non standard read/writes.
//	
//	Arguments:
//	
//		mode	: enum e_CommModes_Generic defined value.
//	    count	: Number of words to be transferred.
//	    addr	: Device addr to access.
//		hostAddr: Host addr where the data resides.
//	

INT32 ReadGeneric
(
	UINT32 mode, UINT32 count, UINT32 addr, UINT32 *hostAddr
)
{
	UINT32 size, dspAddr, offset;
	UINT32 temp = 0;

	if((board.savingQuListFile == 1) && 
		(mode != e_CommModes_Generic_PCI_NVRAM) &&
		(mode != e_CommModes_Generic_PCI_OpReg))
	{
		if (count == 0)
			return e_Err_NoError;

		//set file pointer to biginning
		fseek(board.quListfp, kPLXC33_ParseableDataStart, SEEK_SET);
		//search file for that dsp addr and size
		while(!ferror(board.quListfp))
		{
			//goto and read the count
			fseek(board.quListfp, sizeof(UINT32), SEEK_CUR);
			if(fread(&size, sizeof(UINT32), 1, board.quListfp) != 1)
				break;

			//goto and read the dsp addr
			fseek(board.quListfp, sizeof(UINT32), SEEK_CUR);
			if(fread(&dspAddr, sizeof(UINT32), 1, board.quListfp) != 1)
				break;

			if
			(
				((dspAddr + size * board.boardInfo.cgDSP_DataSize) >=
				(addr + count * board.boardInfo.cgDSP_DataSize)) 
				&& (addr >= dspAddr)
			)
			{	
				offset = addr - dspAddr;	

				temp = (UINT32)ftell(board.quListfp);
				if(offset > 0)
					temp += offset;
			
			}

			fseek(board.quListfp, sizeof(UINT32) * size, SEEK_CUR);
		}

		if(temp)
		{
			fseek(board.quListfp, temp, SEEK_SET);
			if((fread(hostAddr, sizeof(UINT32), count, board.quListfp)) != count)
				return e_Err_FileReadError;
			return e_Err_NoError;
		}
		else
			return e_Err_UnableToReadFile;	
	}
	else
	{
	
		switch(mode)
		{
		case e_CommModes_Generic_PCI_Target:
			return 
				SI_PLX_ReadTarget
				(
					board.cgpSIDev, 
					0, count, addr, hostAddr
				);
			break;

		case e_CommModes_Generic_PCI_PLX_BMDMA:
			return 
				SI_PLX_ReadBlockDMA
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);
			break;

		case e_CommModes_Generic_PCI_Hostpoll:
			return 
				SI_PLXC33_ReadHostpoll
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);
			break;

		case e_CommModes_Generic_PCI_NVRAM:
			return
				SI_PLX_ReadPCI_NVWord
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);				
			break;

		case e_CommModes_Generic_PCI_OpReg:
			return
				SI_PLX_ReadPCI_OpReg
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);				
			break;

		default:
			return e_Err_UnknownCommand;
		}

		return e_Err_NoError;
	}

}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    Non standard read/writes.
//	
//	Arguments:
//	
//		mode	: enum e_CommModes_Generic defined value.
//	    count	: Number of words to be transferred.
//	    addr	: Device addr to access.
//		hostAddr: Host addr where the data resides.
//	

INT32 WriteGeneric
(
	UINT32 mode, UINT32 count, UINT32 addr, UINT32 *hostAddr
)
{
	UINT32 temp = 0;

	board.listSize += count;
	
	if((board.savingQuListFile == 1) && 
		(mode != e_CommModes_Generic_PCI_NVRAM) &&
		(mode != e_CommModes_Generic_PCI_OpReg))
	{
		if (count == 0)
			return e_Err_NoError;

		fseek(board.quListfp, 0, SEEK_END);

		if((fwrite(&board.dataType, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(&count, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(&temp, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(&addr, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(hostAddr, sizeof(UINT32), count, board.quListfp)) != count)
			return e_Err_FileWriteError;
	
		return e_Err_NoError;
	}
	else
	{
		switch(mode)
		{
		case e_CommModes_Generic_PCI_Target:
			return 
				SI_PLX_WriteTarget
				(
					board.cgpSIDev, 
					0, count, addr, hostAddr
				);
			break;

		case e_CommModes_Generic_PCI_PLX_BMDMA:
			return 
				SI_PLX_WriteBlockDMA
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);
			break;

		case e_CommModes_Generic_PCI_Hostpoll:
			return 
				SI_PLXC33_WriteHostpoll
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);
			break;

		case e_CommModes_Generic_PCI_NVRAM:
			return
				SI_PLX_WritePCI_NVWord
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);				
			break;

		case e_CommModes_Generic_PCI_OpReg:
			return
				SI_PLX_WritePCI_OpReg
				(
					board.cgpSIDev, 
					count, addr, hostAddr
				);				
			break;

		default:
			return e_Err_UnknownCommand;
		}

		return e_Err_NoError;
	}
}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    Sets locations where Q-List starts and ends, according to
//		memory available.
//	
//	Arguments:
//	
//		memoryParams: Pointer to Array. C33 uses only a single value, but kept
//			the array form to match for C6711, also, if more parameters are
//			added, they'll be just added into the array.
//	

INT32 CoffLoad_ConfigMemory
(
	UINT32 *memoryParams
)
{

#ifdef vqhost_EXPORTS
	if(memoryParams[1])
	{
			// if Bank 1 exists
			cgDSP_BaseAddrFel	= 0x400000 + kPLXC33_BaseAddrFel;
			cgDSP_BaseAddrPqf	= 0x400000 + kPLXC33_BaseAddrPqf;
			cgDSP_BaseAddrBHead	= 0x400000 + kPLXC33_BaseAddrBHead;
			cgDSP_BaseAddrBlock	= 0x400000 + kPLXC33_BaseAddrBlock;
			cgDSP_MaxBlockAddr	= 0x400000 + memoryParams[0] + memoryParams[1];
	}
	else
	{
			// if Bank 1 doesn't exist
			cgDSP_BaseAddrFel	= kPLXC33_BaseAddrFel;
			cgDSP_BaseAddrPqf	= kPLXC33_BaseAddrPqf;
			cgDSP_BaseAddrBHead	= kPLXC33_BaseAddrBHead;
			cgDSP_BaseAddrBlock	= kPLXC33_BaseAddrBlock;
			cgDSP_MaxBlockAddr	= memoryParams[0];
	}

#endif // vqhost_EXPORTS

/*
	//switch depending on memory size and initialize parameters
	switch(memoryParams[0])
	{
		case 0:	// 128k at PAGE0
			cgDSP_BaseAddrFel	= ckDSP_BaseAddrFel0;
			cgDSP_BaseAddrPqf	= ckDSP_BaseAddrPqf0;
			cgDSP_BaseAddrBHead	= ckDSP_BaseAddrBHead0;
			cgDSP_BaseAddrBlock	= ckDSP_BaseAddrBlock0;
			cgDSP_MaxBlockAddr	= 0x1FFFF;
			break;

		case 1:	// 512k at PAGE0
			cgDSP_BaseAddrFel	= ckDSP_BaseAddrFel0;
			cgDSP_BaseAddrPqf	= ckDSP_BaseAddrPqf0;
			cgDSP_BaseAddrBHead	= ckDSP_BaseAddrBHead0;
			cgDSP_BaseAddrBlock	= ckDSP_BaseAddrBlock0;
			cgDSP_MaxBlockAddr	= 0x7FFFF;
			break;

		case 2: // 128k at PAGE0, 128k at PAGE1		
			cgDSP_BaseAddrFel	= 0x400000 - 0x20000 + ckDSP_BaseAddrFel0;
			cgDSP_BaseAddrPqf	= 0x400000 - 0x20000 + ckDSP_BaseAddrPqf0;
			cgDSP_BaseAddrBHead	= 0x400000 - 0x20000 + ckDSP_BaseAddrBHead0;
			cgDSP_BaseAddrBlock	= 0x400000 - 0x20000 + ckDSP_BaseAddrBlock0;
			cgDSP_MaxBlockAddr	= 0x400000 - 0x20000 + 0x3FFFF;
			break;

		case 3: // 128k at PAGE0, 512k at PAGE1
			cgDSP_BaseAddrFel	= 0x400000 - 0x20000 + ckDSP_BaseAddrFel0;
			cgDSP_BaseAddrPqf	= 0x400000 - 0x20000 + ckDSP_BaseAddrPqf0;
			cgDSP_BaseAddrBHead	= 0x400000 - 0x20000 + ckDSP_BaseAddrBHead0;
			cgDSP_BaseAddrBlock	= 0x400000 - 0x20000 + ckDSP_BaseAddrBlock0;
			cgDSP_MaxBlockAddr	= 0x400000 - 0x20000 + 0x9FFFF;
			break;

		case 4: // 512k at PAGE0, 128k at PAGE1
			cgDSP_BaseAddrFel	= 0x400000 - 0x80000 + ckDSP_BaseAddrFel0;
			cgDSP_BaseAddrPqf	= 0x400000 - 0x80000 + ckDSP_BaseAddrPqf0;
			cgDSP_BaseAddrBHead	= 0x400000 - 0x80000 + ckDSP_BaseAddrBHead0;
			cgDSP_BaseAddrBlock	= 0x400000 - 0x80000 + ckDSP_BaseAddrBlock0;
			cgDSP_MaxBlockAddr	= 0x400000 - 0x80000 + 0x9FFFF;
			break;

		case 5: // 512k at PAGE0, 512k at PAGE1
			cgDSP_BaseAddrFel	= 0x400000 - 0x80000 + ckDSP_BaseAddrFel0;
			cgDSP_BaseAddrPqf	= 0x400000 - 0x80000 + ckDSP_BaseAddrPqf0;
			cgDSP_BaseAddrBHead	= 0x400000 - 0x80000 + ckDSP_BaseAddrBHead0;
			cgDSP_BaseAddrBlock	= 0x400000 - 0x80000 + ckDSP_BaseAddrBlock0;
			cgDSP_MaxBlockAddr	= 0x400000 - 0x80000 + 0xfffff;
			break;

		default:
			return e_Err_BufferTooSmall;	// No Memory Present
	}
*/

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    For C33, Coff load Flash memory from a file
//	
//	Arguments:
//	
//		They should not be molested.
//	

INT32 CoffQuListfromFile(char quListFilename[])
{
	UINT32 cnt,count, *dspMem, error;
	
	//Coffload from created file
	//	Put board into reset.
	error = SI_PLXC33_ResetBoard( board.cgpSIDev, e_Reset_Assert );
	if ( error != e_Err_NoError )
		return error;
		
	// malloc for dsp memory
	dspMem = (UINT32 *)malloc(kCOFFFile_MaxBufferSize);
	if (dspMem == NULL)
		return e_Err_WindowsMemError;

	board.quListfp = fopen(quListFilename, "rb");
	if ( board.quListfp == NULL )
	{
		free(dspMem);
		return e_Err_FileNotFound;
	}

	// read QuList file, count Bytes, into dspMem (UINT32)
	count = fread(dspMem, sizeof(UINT32), kCOFFFile_MaxBufferSize / sizeof(UINT32), board.quListfp);
	fclose(board.quListfp);	

	board.quListFileSize = count;
	
	// load all sections from COFF file, as a single write.
	error = SI_PLXC33_WriteTarget
			(
				board.cgpSIDev,
				0,
				count,
				kPLXC33_CoffLoadOffset,
				dspMem
			);
		if( error != e_Err_NoError )
		{
			return error;
		}

	// write reset vector to CoffFileOffset so when reset is removed it
	// starts our si_c_int00 routine, parsing cofffile to correct locations
	cnt = kPLXC33_CoffLoadOffset;
	error = SI_PLXC33_WriteTarget
			(
				board.cgpSIDev,
				0,
				1,
				kPLXC33_ResetVectorAddr,
				&cnt
			);
	if( error != e_Err_NoError )
	{
		return error;
	}
	
	free(dspMem);

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	
//	Routine Description:
//	
//	    For C33, Close QuList file, Coff load Flash memory from QuList file,
//		and Launch.
//	
//	Arguments:
//	
//		They should not be molested.
//	

INT32 CloseQuListFileandLaunch(char quListFilename[])
{
	UINT32 endTag;
	INT32 error;

	//close QuList File
	if(board.quListfp)
	{
		// Addr and size of 0xffffffff means this is the end.
		endTag = 0xffffffff;

		fseek(board.quListfp, 0, SEEK_END);

		//have to write it twice
		if((fwrite(&endTag, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;
		if((fwrite(&endTag, sizeof(UINT32), 1, board.quListfp)) != 1)
			return e_Err_FileWriteError;

		fclose(board.quListfp);
		board.quListfp = NULL;
		board.savingQuListFile = 0;

		error = CoffQuListfromFile(quListFilename);
		if (error) return error;

		error = SI_PLXC33_ResetBoard( board.cgpSIDev, e_Reset_Toggle );
		if (error) return error;
	}
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
