///////////////////////////////////////////////////////////////////////////////
//	sisamplib.cpp
//
//	Description:
//		Main entry point for the DLL sample program for WDM driver.
//		The main function of this file is to open the driver to obtain the 
//		handle and pass on the control to the main loop in sisamplibf.cpp.
//
//	Revision History:
//		2002-03-07: mik
//			Created.
//		2002-04-18: mik
//			Moved rudimentary addon-initiated to sicommon_plx.xxx.
//			Redefined target/bmdma (only byte boundary).
//			Restructured so that it's easy to move from sisample to sample.
//		2002-06-21: mik
//			Fixed printf left by Av or Ig. (who modified it last?)
//			Made it work with C6711 using new comm modes.
//		2002-06-28: mik
//			Changed DSP Active comm for C6711 to use DSP Active Comm API
//			calls using new comm modes.
//		2002-07-01: mik
//			Changed all instances of SIHW_ to SI_
//		2002-07-03: mik
//			Modified to use SI_PLXC6711_OpenDriver.
//		2003-06-19: Mik
//			Using extra params for COFF load/verify.
//		2003-07-09: mik
//			Coffload/verify returns DRAM parameters.
//		2006-04-09: Whipple
//			Changed CoffLoad to work with new options
//			and moved reset into the DLL(from the vi)
//
///////////////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include "../../../../../common_sihw/cboard.h"

#include "sisamplib.h"

///////////////////////////////////////////////////////////////////////////////
// globals 

bool gStarted = false;

CPLXC33	gCPLXBoard[4];

// board will contain a CPLXC33 or a CPLXC6711, depending which is it compiled for
CPLXBoard *board[4] = {&gCPLXBoard[0], &gCPLXBoard[1], &gCPLXBoard[2], &gCPLXBoard[3]};


///////////////////////////////////////////////////////////////////////////////
INT32 SI_PLX_SetupCallbackEvent(PPLXDevice pPLXDev, void *callbackFunction);

void _init()
{
	SI_DLL_Entry();
}

void _fini()
{
	SI_DLL_Exit();
}

void callback0(int param)
{
	static int trigger = 0;

	++trigger;
	gCPLXBoard[0].cgpSIDev->wait_event = 1;
}
void callback1(int param)
{
	static int trigger = 0;

	++trigger;
	
	gCPLXBoard[1].cgpSIDev->wait_event = 1;
}
void callback2(int param)
{
	static int trigger = 0;

	++trigger;
	
	gCPLXBoard[2].cgpSIDev->wait_event = 1;
}
void callback3(int param)
{
	static int trigger = 0;

	++trigger;
	
	gCPLXBoard[3].cgpSIDev->wait_event = 1;
}

///////////////////////////////////////////////////////////////////////////////

bool SI_DLL_Entry()
{
	UINT32 boardID, error;
	char message[256];
			 
	if (gStarted)
		return true;

	// only do this code when the 1st process attaches
	gStarted = true;

	// clear message file before starting
	OS_Popup_MessageBox("", "", 0);	// clear 

	// check for up to four boards
	for(boardID = 0; boardID < 4; boardID++)
	{

#ifdef SIDEF_PLX_C3xC671x	// For both C33 & C6711, class will be created accordingly.
		error = board[boardID]->ClassLoad( &boardID );
		if (error != e_Err_NoError)
		{
			break;
		}
#endif //SIDEF_PLX_C3xC671x
		
		if ( board[boardID]->cgpSIDev->addonInitBufferSize == 0)
		{
			OS_Popup_MessageBox
			(
				"Driver reported an error allocating addon-init Buffer\n"
				"Interrupt based Add-on-initiated will not function.", 
				"sisamplib", 1
			);
		}
		else
		{
			sprintf
			(
				message, "%d%s", 
				board[boardID]->cgpSIDev->addonInitBufferSize, 
				" UINT32 allocated for add-on-initiated buffer\n"
			);
			OS_Popup_MessageBox( message, "sisamplib", 1);
		}

		switch(boardID)
		{
		case 0:
			error = SI_PLX_SetupCallbackEvent( gCPLXBoard[boardID].cgpSIDev, (void *)callback0 );
			if (error != e_Err_NoError)
				return error;
			break;
		case 1:
			error = SI_PLX_SetupCallbackEvent( gCPLXBoard[boardID].cgpSIDev, (void *)callback1 );
			if (error != e_Err_NoError)
				return error;
			break;
		case 2:
			error = SI_PLX_SetupCallbackEvent( gCPLXBoard[boardID].cgpSIDev, (void *)callback2 );
			if (error != e_Err_NoError)
				return error;
			break;
		case 3:
			error = SI_PLX_SetupCallbackEvent( gCPLXBoard[boardID].cgpSIDev, (void *)callback3 );
			if (error != e_Err_NoError)
				return error;
			break;
		}		
	} // end for loop

	if (boardID == 0)
	{
		OS_Popup_MessageBox( "No boards found.", "sisamplib", 1 );
		gStarted = false;
		return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////

INT32 SI_DLL_Exit()
{
	int boardID;

	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	for(boardID = 0; boardID < 4; boardID++)
		// When done, close the handle to the driver
//		SI_PLX_CloseDriver( board[boardID]->cgpSIDev );
		board[boardID]->ClassUnload();

	return true;   // successful
}

///////////////////////////////////////////////////////////////////////////////


DLLReturnType  
Sample_WriteMailboxRegister
(
	INT32 boardID, 
	UINT32 mailboxNum, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 		
		SI_PLX_WriteMailbox
		(
			board[boardID]->cgpSIDev, 
			1, mailboxNum, data
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ReadMailboxRegister
(
	INT32 boardID, 
	UINT32 mailboxNum, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_ReadMailbox
		(
			board[boardID]->cgpSIDev, 
			1, mailboxNum, data
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ReadConfigurationSpace
(
	INT32 boardID, 
	UINT32 count, 
	UINT32 offset, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_ReadPCI_ConfSpace
		(
			board[boardID]->cgpSIDev, 
			count, offset, data
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_WriteOpReg
(
	INT32 boardID, 
	UINT32 opRegNum, 
	UINT32 *data
)
{	
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_WritePCI_OpReg
		(
			board[boardID]->cgpSIDev, 
			1, opRegNum, data
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ReadOpReg
(
	INT32 boardID, 
	UINT32 opRegNum, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;
	
	return 
		SI_PLX_ReadPCI_OpReg
		(
			board[boardID]->cgpSIDev, 
			1, opRegNum, data
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_WriteBusmastered
(
	INT32 boardID, 
	UINT32 count, 
	UINT32 offset, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	// it is 24bit DWORD offset Address as seen by C33 DSP
	return 
		SI_PLXC33_WriteBlockDMA
		(
			board[boardID]->cgpSIDev, 
			count, offset, data 
		);

}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ReadBusmastered
(
	INT32 boardID, 
	UINT32 count, 
	UINT32 offset, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	// it is 24bit DWORD offset Address as seen by C33 DSP
	return 
		SI_PLXC33_ReadBlockDMA
		(
			board[boardID]->cgpSIDev, 
			count, offset, data
		);

}

DLLReturnType  
Sample_WriteTarget
(
	INT32 boardID, 
	UINT32 region,
	UINT32 count, 
	UINT32 offset,
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	// it is 24bit DWORD offset Address as seen by C33 DSP
	return 
		SI_PLXC33_WriteTarget
		(
			board[boardID]->cgpSIDev, 
			region, count, offset, data			
		);
}

///////////////////////////////////////////////////////////////////////////////
DLLReturnType  
Sample_ReadTarget
(
	INT32 boardID, 
	UINT32 region, 
	UINT32 count, 
	UINT32 offset,
	UINT32 *data
)
{

	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	// it is 24bit DWORD offset Address as seen by C33 DSP
	return
		SI_PLXC33_ReadTarget
		(
			board[boardID]->cgpSIDev, 
			region, count, offset, data
		);

}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_WriteNVRAM
(
	INT32 boardID, 
	UINT32 addrNVRam, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_WritePCI_NVWord
		(
			board[boardID]->cgpSIDev, 
			1, addrNVRam, data
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ReadNVRAM
(
	INT32 boardID, 
	UINT32 addrNVRam, 
	UINT32 *data
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_ReadPCI_NVWord
		(
			board[boardID]->cgpSIDev, 
			1, addrNVRam, data
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_SetTimeout
(
	INT32 boardID, 
	UINT32 milliseconds
)
{	
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_SetTimeout
		(
			board[boardID]->cgpSIDev, 
			milliseconds
		);
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_CancelBusmastering
(
	INT32 boardID
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_CancelBlockDMA( board[boardID]->cgpSIDev );
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_GetBaddr
(
	INT32 boardID, 
	UINT32 *dirAddr[]
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_DirectAccessAddr( board[boardID]->cgpSIDev, 6, dirAddr );
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ReleaseBaddr
(
	INT32 boardID
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_ReleaseDirectAccessAddr( board[boardID]->cgpSIDev );
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_DriverConfigPointBlock
(
	INT32 boardID, UINT32 pointOrBlock
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return 
		SI_PLX_DriverConfig_BlockPoint( board[boardID]->cgpSIDev, pointOrBlock );	
}

//////////////////////////////////////////////////////////////////////////////
// Begin: For sample program demo only.
///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_TranslateErrorCode(int error, char errorMsg[])
{
	return TranslateErrorCode(error, errorMsg);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//	Following are specific to SI board. Do not use them for non SI boards.
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Begin: Used only by SI boards.

#ifdef SIDEF_PLX_C3xC671x

DLLReturnType  
Sample_ResetBoard
(
	INT32 boardID,
	UINT32 resetMode
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return
		SI_PLXC33_ResetBoard( board[boardID]->cgpSIDev, resetMode	);

}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_LoadCofffile
(
	INT32 boardID, 
	char filename[], 
	UINT32 coffOptions
)
{
	INT32 error;

	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	if(coffOptions != 1)	// != 1, means default
	{
		// load coff file for C33
		UINT32 	*beforeCoff, beforeCoffCount;
		UINT32 memoryParams[2];			// Use array to match C6711 SPD data format
		UINT32 	dspMemCount = kCOFFImage_MaxBufferSize / sizeof(UINT32),
				*passiveCoffImage, passiveCoffImageCount,
				*activeCoffImage, activeCoffImageCount, cnt;

		beforeCoff = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
		if (beforeCoff == NULL)
			return e_Err_WindowsMemError;

		error = SI_PLXC33_CoffLoad_GetMemoryParams(board[boardID]->cgpSIDev, memoryParams);
		if(error) return error;

		board[boardID]->pBoardInfo->cgDSP_Bank0 = memoryParams[0];
		board[boardID]->pBoardInfo->cgDSP_Bank1 = memoryParams[1];

		// parameters to write before COFF file is written
		beforeCoffCount = 1;	// reserve first for count

		// if something must be added before Coff file, add it here and
		// increment beforCoffCount++ per each item, Follow following example
		// beforeCoff[beforeCoffCount++] = 0;			//type

		// actual number of words is one less than total (first is count)
		beforeCoffCount --;
		beforeCoff[0] = beforeCoffCount;

		passiveCoffImage = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
		if (passiveCoffImage == NULL)
		{
			free(beforeCoff);
			return e_Err_WindowsMemError;
		}

		activeCoffImage = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
		if (activeCoffImage == NULL)
		{
			free(beforeCoff);
			free(passiveCoffImage);
			return e_Err_WindowsMemError;
		}

		for(cnt = 0; cnt < dspMemCount; cnt++)
		{
			passiveCoffImage[cnt] = 0;
			activeCoffImage[cnt] = 0;
		}

		error = SI_PLXC33_ParseCoff(	
			board[boardID]->cgpSIDev, filename, dspMemCount,
			passiveCoffImage, &passiveCoffImageCount,
			activeCoffImage, &activeCoffImageCount,
			beforeCoff);
		if(error != e_Err_NoError)
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}

	// Start Passive Coff Load
		error = 
			SI_PLXC33_CoffWrite
			(
				board[boardID]->cgpSIDev, filename, 0,//coffOptions, 
				passiveCoffImage, passiveCoffImageCount
			);
		if ( error != e_Err_NoError )
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}

		// verify coff file
		error = 
			SI_PLXC33_CoffVerify
			(
				board[boardID]->cgpSIDev, filename, 0,//coffOptions, 
				passiveCoffImage, passiveCoffImageCount
			);
		if ( error != e_Err_NoError )
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}
	// End Passive Coff Load;

		// 3. User program should toggle reset. New code is loaded and ready, but 
		//		not yet running until the reset is toggled.
		error = 
			SI_PLXC33_ResetBoard( board[boardID]->cgpSIDev, e_Reset_Toggle );
		if ( error != e_Err_NoError )
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}

	// Start Active Coff Load
		error = 
			SI_PLXC33_CoffWrite
			(
				board[boardID]->cgpSIDev, filename, 1,//coffOptions, 
				activeCoffImage, activeCoffImageCount
			);
		if ( error != e_Err_NoError )
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}

		// verify coff file
		error = 
			SI_PLXC33_CoffVerify
			(
				board[boardID]->cgpSIDev, filename, 1,//coffOptions, 
				activeCoffImage, activeCoffImageCount
			);
		if ( error != e_Err_NoError )
		{
			free(beforeCoff);
			free(passiveCoffImage);
			free(activeCoffImage);
			return error;
		}

		free(beforeCoff);
		free(passiveCoffImage);
		free(activeCoffImage);

	// End Active Coff Load

	}
	else
	{
		error = board[boardID]->CoffQuListfromFile(filename);
		if(error)
		{
			OS_Popup_ErrorMessageBox
			(
				"Error loading COFF+QuList file\n",
				"Loading COFF+QuList file",
				error,
				0
			);
			return error;
		}

		error = board[boardID]->ResetBoard( e_Reset_Toggle );
		if (error)
		{
			OS_Popup_ErrorMessageBox
			(
				"Error reseting board\n",
				"Loading COFF+QuList file",
				error,
				0
			);
			return error;
		}
	}

	return error;
}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ReadDSPActiveComm
(
	INT32 boardID, 
	UINT32 count, 
	UINT32 offset, 
	UINT32 *hostAddr, 
	UINT32 mode
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	mode = mode + 2;		// modes 0 & 1 in C33 are passive.

	return
		SI_PLXC33_ReadDSPActive
		(
			board[boardID]->cgpSIDev, 
			mode, count, offset, hostAddr
		);

}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_WriteDSPActiveComm
(
	INT32 boardID, 
	UINT32 count, 
	UINT32 offset, 
	UINT32 *hostAddr, 
	UINT32 mode
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	mode = mode + 2;		// modes 0 & 1 in C33 are passive.

	return
		SI_PLXC33_WriteDSPActive
		(
			board[boardID]->cgpSIDev, 
			mode, count, offset, hostAddr
		);

}

///////////////////////////////////////////////////////////////////////////////

DLLReturnType  
Sample_ConfigureFPGA
(
	INT32 boardID,
	char filename[]
)
{
	INT32 error;

	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	error =
		LoadBoardFPGAs
		(
			boardID, board[boardID]->cgpSIDev, filename
		);
	if (error != 0)
	{
		return e_Err_MiscError;
	}

	return e_Err_NoError;;
}


DLLReturnType  
Sample_CONV_FLOAT_DSP2HOST
(
	INT32 boardID,
	UINT32 size,
	UINT32 *dataIn,				// DSP format
	float *dataOut				// Host format.
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return board[boardID]->Conv_Float_DSP2Host( size, dataIn, dataOut );
}


DLLReturnType  
Sample_CONV_FLOAT_HOST2DSP
(
	INT32 boardID,
	UINT32 size,
	float *dataIn,				// Host format
	UINT32 *dataOut				// DSP format.
)
{
	// windows is guaranteed to have it already started, linux is not.
	if (!gStarted)
		if (! SI_DLL_Entry() )
			return e_Err_DeviceNotFound;

	return board[boardID]->Conv_Float_Host2DSP( size, dataIn, dataOut );
}



#endif	//SIDEF_PLX_C3xC671x

// End: Used only by SI boards.
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////











