// VQ_addTest.cpp : Defines the entry point for the DLL application.
//

#include "stdafx.h"
#include "vq_addtest.h"

// Declare pointer to sisamplib in order to load it.
HINSTANCE__ *hMod;

// Declare some C33 constants.
UINT32 const kMaxFileSize = 0x00400000;		//max. file size = 4MB.
UINT32 kSI_C33CoffLoadAddress = 0x00010000;	//Coff Load Address.
UINT32 kSI_C33ResetVector = 0x00000000;		//Reset Vector Address.


// Declarations of structures matching the structure to be
// used in order to call each function from sisamplib.dll
typedef (CALLBACK *Sample_PLXC33_WriteTarget_t)
(
	INT32 board, 
	UINT32 region,
	UINT32 count, 
	UINT32 offset, 
	UINT32 *data
);


typedef (CALLBACK *Sample_PLXC33_ReadTarget_t)
(
	INT32 board, 
	UINT32 region, 
	UINT32 count, 
	UINT32 offset, 
	UINT32 *data
);

typedef  (CALLBACK *Sample_CONV_FLOAT_HOST2DSP_t)
(
	INT32 boardID,
	UINT32 sizeOfInput,
	float *inputValues,
	UINT32 *tempUINT32
);


typedef (CALLBACK *Sample_CONV_FLOAT_DSP2HOST_t)
(
	INT32 boardID,
	INT32 size,
	UINT32 *dataIn,				// DSP format
	float *dataOut				// Host format.
);

typedef (CALLBACK *Sample_ResetBoard_t)
(
	INT32 boardID,
	UINT32 resetMode
);

typedef (CALLBACK *Sample_LoadCofffile_t)
(
	INT32 boardID,
	char coffFileName[],
	UINT32 coffOptions
);

// Declare pointer to previous structures, which
// will be converted into pointers to functions
// from sisamplib.dll
Sample_PLXC33_WriteTarget_t		Sample_PLXC33_WriteTarget;
Sample_PLXC33_ReadTarget_t		Sample_PLXC33_ReadTarget;
Sample_CONV_FLOAT_HOST2DSP_t	Sample_CONV_FLOAT_HOST2DSP;
Sample_CONV_FLOAT_DSP2HOST_t	Sample_CONV_FLOAT_DSP2HOST;
Sample_ResetBoard_t				Sample_ResetBoard;
Sample_LoadCofffile_t			Sample_LoadCofffile;


BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{

    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:

			loadSISamplibDll();

			break;
		case DLL_THREAD_ATTACH:

			break;
		case DLL_THREAD_DETACH:

			break;
		case DLL_PROCESS_DETACH:

			unloadSISamplibDll();

			break;
	}

	return TRUE;
}


// Routine which loads sisamplib.dll and obtain pointers to all needed functions.
// Note that location of sisamplib.dll is hard coded here.
// User must change it appropiately.
DllExport INT32 WINAPI
loadSISamplibDll()
{
	int errorCode;

		// Open sisamplib.dll.
		hMod = LoadLibrary ("sisamplib.dll");
		errorCode = GetLastError();
		if(!hMod)
			return eErr_LoadLibrary;

		// Obtaining pointers to functions inside sisamplib.dll.
		Sample_PLXC33_WriteTarget = (Sample_PLXC33_WriteTarget_t)
			GetProcAddress(hMod, "Sample_PLXC33_WriteTarget");
		errorCode = GetLastError();
		if(!Sample_PLXC33_WriteTarget)
			return eErr_ObtainPointertoFunction;

		Sample_PLXC33_ReadTarget = (Sample_PLXC33_ReadTarget_t)
			GetProcAddress(hMod, "Sample_PLXC33_ReadTarget");
		errorCode = GetLastError();
		if(!Sample_PLXC33_ReadTarget)
			return eErr_ObtainPointertoFunction;

		Sample_CONV_FLOAT_HOST2DSP = (Sample_CONV_FLOAT_HOST2DSP_t)
			GetProcAddress(hMod, "Sample_CONV_FLOAT_HOST2DSP");
		errorCode = GetLastError();
		if(!Sample_CONV_FLOAT_HOST2DSP)
			return eErr_ObtainPointertoFunction;

		Sample_CONV_FLOAT_DSP2HOST = (Sample_CONV_FLOAT_DSP2HOST_t)
			GetProcAddress(hMod, "Sample_CONV_FLOAT_DSP2HOST");
		errorCode = GetLastError();
		if(!Sample_CONV_FLOAT_DSP2HOST)
			return eErr_ObtainPointertoFunction;

		Sample_ResetBoard = (Sample_ResetBoard_t)
			GetProcAddress(hMod, "Sample_ResetBoard");
		errorCode = GetLastError();
		if(!Sample_ResetBoard)
			return eErr_ObtainPointertoFunction;

		Sample_LoadCofffile = (Sample_LoadCofffile_t)
			GetProcAddress(hMod, "Sample_LoadCofffile");
		errorCode = GetLastError();
		if(!Sample_ResetBoard)
			return 7;

	return 0;
}


// Routine to unloads sisamplib.dll.
DllExport INT32 WINAPI
unloadSISamplibDll()
{
	FreeLibrary(hMod);

	return 0;
}


// Coffloads DSP with a complete coff file with QuList.
DllExport INT32 WINAPI
VQ_addTest_LoadQuListFile
(
	INT32 boardID,
	char qulistfilepath[]
)
{
	FILE *filePointer;
	UINT32 count;
	INT32 error;
	char *fileData;

	
	fileData = (char *) malloc(kMaxFileSize);
	error = GetLastError();
	if(error)
		return error;

	// Open QuListFile.
	filePointer = fopen(qulistfilepath, "rb");
	error = GetLastError();
	if(error)
		return error;
	if(!filePointer)
		return eErr_ObtainPointertoFile;

	// Read QuListFile into fileData.
	count = fread(fileData, sizeof(char), kMaxFileSize, filePointer);
	error = GetLastError();
	if(error)
		return error;

	// Close QuListFile.
	fclose(filePointer);

	// Set Board to Reset;
	error = Sample_ResetBoard
	(
		boardID,
		0
	);
	if(error)
		return error;

	// Load Coff File.
	error = Sample_PLXC33_WriteTarget
	(
		boardID, 
		0, 
		count / sizeof(UINT32), 
		kSI_C33CoffLoadAddress,
		(UINT32 *) fileData
	);
	if(error)
		return error;

	free(fileData);

	// Load Reset Vector for C33, must point to where Coff File was loaded.
	// Reset Vector is located at address 0x00000000 in the C33.
	error = Sample_PLXC33_WriteTarget
	(
		boardID, 
		0, 
		1, 
		kSI_C33ResetVector, 
		&kSI_C33CoffLoadAddress
	);
	if(error)
		return error;

	// Remove Reset from Board;
	error = Sample_ResetBoard
	(
		boardID,
		1
	);
	if(error)
		return error;

	return 0;
}


// Manage Inputs. Converts values from Host to DSP formats,
// then write them into the DSP.
DllExport INT32 WINAPI
VQ_addTest_manageInputs
(
	INT32 boardID,
	UINT32 *inputHeaders,
	UINT32 numOfHeaders,
	float *inputValues,		// user must create the array of proper size before calling.
	UINT32 sizeOfValues,
	UINT32 *numHeadersWritten,
	UINT32 *numDataWritten
)
{
	UINT32 *tempUINT32, headercnt, datacnt, error;
	DataHeader tempHeader;

	tempUINT32 = (UINT32 *) malloc(sizeOfValues * sizeof(UINT32));

	// Convert numbers from Host to DSP format.
	error = Sample_CONV_FLOAT_HOST2DSP
	(
		boardID,
		sizeOfValues,
		inputValues,			// Host format.
		tempUINT32				// DSP format
	);
	if(error)
	{
		free(tempUINT32);
		return error;
	}

	*numHeadersWritten = 0;
	datacnt = sizeOfValues;
	*numDataWritten = 0;
	for(headercnt = 0; headercnt < numOfHeaders; headercnt++)
	{
		// read a header
		error = Sample_PLXC33_ReadTarget
		(
			boardID, 
			0, 
			HeaderSize,
			inputHeaders[headercnt], 
			(UINT32 *) &tempHeader
		);
		if(error)
		{
			free(tempUINT32);
			return error;
		}
		if(tempHeader.M * tempHeader.N > datacnt)
		{
			free(tempUINT32);
			return eErr_DataSize;
		}

		// Write values block by block into DSP.
		error = Sample_PLXC33_WriteTarget
		(
			boardID, 
			0, 
			tempHeader.M * tempHeader.N, 
			tempHeader.BLOCK_ADDR, 
			&tempUINT32[sizeOfValues - datacnt]
		);
		if(error)
		{
			free(tempUINT32);
			return error;
		}

		// Update generation
		error = Sample_PLXC33_WriteTarget
		(
			boardID,
			0,
			1,
			inputHeaders[headercnt],
			&gGeneration
		);
		if(error)
		{
			free(tempUINT32);
			return error;
		}

		(*numHeadersWritten)++;
		datacnt = datacnt - tempHeader.M * tempHeader.N;
		*numDataWritten = sizeOfValues - datacnt;
	}
	gGeneration++;

	free(tempUINT32);

	return 0;
}


// Manage Outputs. Read values from DSP,
// then convert them from DSP to Host format.
DllExport INT32 WINAPI
VQ_addTest_manageOutputs
(
	INT32 boardID,
	UINT32 *outputHeaders,
	UINT32 numOfHeaders,
	float *outputValues,	// user must create the array of proper size before calling.
	UINT32 sizeOfValues,
	UINT32 *numHeadersRead,
	UINT32 *numDataRead,
	UINT32 synctoGeneration
)
{
	UINT32 *tempUINT32, tempGeneration, headercnt, datacnt, error;
	DataHeader tempHeader;

	tempUINT32 = (UINT32 *) malloc(sizeOfValues * sizeof(UINT32));

	*numHeadersRead = 0;
	datacnt = sizeOfValues;
	*numDataRead = 0;
	for(headercnt = 0; headercnt < numOfHeaders; headercnt++)
	{
		// read a header
		error = Sample_PLXC33_ReadTarget
		(
			boardID, 
			0, 
			HeaderSize,
			outputHeaders[headercnt], 
			(UINT32 *) &tempHeader
		);
		if(error)
		{
			free(tempUINT32);
			return error;
		}

		if(synctoGeneration)
		{
			tempGeneration = tempHeader.GEN;
			while(tempGeneration == tempHeader.GEN)
			{
				// read a header
				error = Sample_PLXC33_ReadTarget
				(
					boardID, 
					0, 
					HeaderSize,
					outputHeaders[headercnt], 
					(UINT32 *) &tempHeader
				);
				if(error)
				{
					free(tempUINT32);
					return error;
				}
			}
		}
		if(tempHeader.M * tempHeader.N > datacnt)
		{
			free(tempUINT32);
			return eErr_DataSize;
		}

		// Read values block by block from DSP.
		error = Sample_PLXC33_ReadTarget
		(
			boardID, 
			0, 
			tempHeader.M * tempHeader.N, 
			tempHeader.BLOCK_ADDR, 
			&tempUINT32[sizeOfValues - datacnt]
		);
		if(error)
		{
			free(tempUINT32);
			return error;
		}

		(*numHeadersRead)++;
		datacnt = datacnt - tempHeader.M * tempHeader.N;
		*numDataRead = sizeOfValues - datacnt;
	}

	// Convert numbers from DSP to Host format.
	Sample_CONV_FLOAT_DSP2HOST
	(
		boardID,
		sizeOfValues - datacnt,	// Convert only what was read.
		tempUINT32,				// DSP format
		outputValues			// Host format.
	);

	free(tempUINT32);

	return 0;
}


// Write inputs into DSP, then read outputs.
DllExport INT32 WINAPI
VQ_addTest_manageInputsAndOutputs
(
	INT32 boardID,
	UINT32 *inputHeaders,
	UINT32 numInputHeaders,
	float *inputValues,		// user must create the array of proper size before calling.
	UINT32 sizeInputValues,
	UINT32 *numHeadersWritten,
	UINT32 *numDataWritten,
	UINT32 *outputHeaders,
	UINT32 numOutputHeaders,
	float *outputValues,	// user must create the array of proper size before calling.
	UINT32 sizeOutputValues,
	UINT32 *numHeadersRead,
	UINT32 *numDataRead,
	UINT32 synctoGeneration
)
{
	INT32 error;

	error = VQ_addTest_manageInputs
	(
		boardID,
		inputHeaders,
		numInputHeaders,
		inputValues,		// user must create the array of proper size before calling.
		sizeInputValues,
		numHeadersWritten,
		numDataWritten
	);
	if(error)
		return error;

	error = VQ_addTest_manageOutputs
	(
		boardID,
		outputHeaders,
		numOutputHeaders,
		outputValues,	// user must create the array of proper size before calling.
		sizeOutputValues,
		numHeadersRead,
		numDataRead,
		synctoGeneration
	);
	if(error)
		return error;

	return 0;
}


DllExport INT32 WINAPI
addTest_LoadCofffile
(
	INT32 boardID,
	char coffFileName[],
	UINT32 coffOptions
)
{
	INT32 error;

	error = Sample_LoadCofffile
	(
		boardID,
		coffFileName,
		coffOptions
	);
	if(error)
		return error;

	return 0;
}
