///////////////////////////////////////////////////////////////////////////////
//	ConfigFPGA_PLX9054.cpp
//
//	Description:
//		Configures the FPGA from BIT file without using the SPROM.
//
//	Revision History:
//		2002-03-04: mik
//			Created
//		2002-04-12: mik
//			Added FPGALoad parameter for indicator
//		2002-04-17: mik
//			Shuffled #define paths to reflect directory changes.
//		2002-07-01: mik
//			Changed all instances of SIHW_ to SI_
//		2002-09-25: mik
//			Code is now platform neutral (only using ANSI code).
//		2003-06-04: mik
//			Changed static allocation in stack to dynamic alloc in heap.
//		2003-08-26: mik
//			Modularize all funcs to support cascaded FPGA.
//		2004-02-23: Ley
//			Added #ifdef DOS for DOS version
//
///////////////////////////////////////////////////////////////////////////////

#include "sihw_common_plx.h"	// sheldon board specific defines

#include "../../../../../common_ddk/os_lnx.h"
//#include "../c33/common/sicommon_plxc33.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define kFPGA_ParamStart	2
#define kFPGA_ParamsPerFPGA	2
#define kFPGA_Param_Type	0
#define kFPGA_Param_Size	1

// 1MBytes is max bit file size
#define kMaxBitFileSize	0x100000	

// up to this many can precede actual bit data
#define kConfigFPGA_ConcatMagicCountMax	256

struct type_ConfigFPGA
{
	struct 
	{
		char *typeString;
		//Length is in bytes, Xilinx Datasheets give this number in bits.
		UINT32 length;

		// magic number when concatenating bit files to load
		// multiple FPGAs connected as Daisy Chain. Varies
		// depending on model of FPGA.
		UINT32 concatMagicCount;
		char concatMagic[kConfigFPGA_ConcatMagicCountMax];
	}fpgaTypes[100];	// up to 100 different FPGAs are supported
	UINT32 totalFPGAAvailble;

	// following are setup on open
	char *buffer;
	UINT32 bufferSize;
	struct PLXDevice *pPLXDev;
	UINT32 cntrlRegValue;

	// following are changing per file
	UINT32 byteStartPosition, byteCount, fpgaTypeIndex;

};

struct type_ConfigFPGA gConfigFPGA;

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_ParseBitFile
//
//	Description: Parse Bit File
//
//	Parameters:
//		char *filename:				Complete path to Bit File.
//		char *buffer:				Buffer to hold contents of Bit File.
//		char *fpgaPartNum:				Name of the FPGA
//		UINT32 bufferSize:			Buffer size.
//		UINT32 *byteStartPosition:	Byte start position.
//		UINT32 *byteCount:			Byte count.
//

INT32 SI_ConfigFPGA_ParseBitFile
(
	char *filename, char *fpgaPartNum, char *buffer, UINT32 bufferSize, 
	UINT32 *byteStartPosition, UINT32 *byteCount
)
{
	FILE *fp;
	UINT32 bufferPtr, cnt, bytesRead, strSize;
	char tempfpgaPartNum1[256], tempfpgaPartNum2[256];
	// hardcoded header values that must be present at the begining of a Xilinx bit file.
	unsigned char
		defaultHeader[] = 
		{
			0x00, 0x09, 0x0f, 0xf0, 0x0f, 0xf0, 0x0f, 0xf0, 
			0x0f, 0xf0, 0x00, 0x00, 0x01
		}, 
		section;

	// first, read the whole file
	fp = fopen( filename, "rb" );
	if(fp == NULL) 
		return e_Err_FileNotFound;

	bytesRead = fread(buffer, sizeof(char), bufferSize, fp);
	fclose(fp);

	// check if filesize is too large
	if (bytesRead == bufferSize)						
		return e_Err_BufferTooSmall;

	// check if default header match
	if (bytesRead < (sizeof(defaultHeader) + 4))									
		return e_Err_InvalidFile;

	for (cnt=0; cnt<sizeof(defaultHeader); cnt++)
		if (buffer[cnt] != (char)defaultHeader[cnt])
			return e_Err_InvalidFile;
	
	// skip over sections until length is 0
	bufferPtr = sizeof(defaultHeader);
	section = 'a';											
	while (1)
	{
		if (buffer[bufferPtr++] != section)
			return e_Err_InvalidFile;

		// actual data starts at 'e'
		if ( (cnt == 0) || (section == 'e') )
			break;

		// verify FPGA part number in bit file versus that outlined in command txt file
		if (section == 'b')
		{
			//Note: bufferPtr + 2 because of '\0' terminating string
			strcpy(tempfpgaPartNum1, (char *)(&buffer[bufferPtr + 2]));
			strSize = strlen(fpgaPartNum);
			strcpy(tempfpgaPartNum2, fpgaPartNum);
			//convert all uppercase to lowercase
			for(cnt = 0; cnt < strSize; cnt++)
			{
				if((tempfpgaPartNum2[cnt] >= 0x41) && (tempfpgaPartNum2[cnt] <= 0x5A))
					tempfpgaPartNum2[cnt] += 0x20;
			}
			//Note: [2] and strSize - 2 is to eliminate first 2 letters from command txt file.
			if(strncmp(&tempfpgaPartNum2[2], tempfpgaPartNum1, strSize - 2))
				return e_Err_FPGA_TypeNotCorrect;
		}

		// this is section size. skip over it.
		cnt = (buffer[bufferPtr] << 8	) 
			+ (buffer[bufferPtr + 1]	);

		bufferPtr += cnt + 2;	// +2 is for size

		if (bufferPtr > bytesRead)
			return e_Err_InvalidFile;

		section++;
	}

	// byte count is just after 'e'. convert to little endian for Intel
	*byteCount = 0;
	*byteCount = ( *byteCount << 8 ) | (unsigned char)buffer[bufferPtr++];
	*byteCount = ( *byteCount << 8 ) | (unsigned char)buffer[bufferPtr++];
	*byteCount = ( *byteCount << 8 ) | (unsigned char)buffer[bufferPtr++];
	*byteCount = ( *byteCount << 8 ) | (unsigned char)buffer[bufferPtr++];

	// byte start position is just after byteCount
	*byteStartPosition = bufferPtr;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_WriteCCLK
//
//	Description: Writes the clock bit to drive it Hi and Lo.
//
//	Parameters:
//		UINT32 bit:		Either an "1" or "0" to be written.
//

INT32 SI_ConfigFPGA_WriteCCLK(UINT32 bit)
{
	if (bit == 0)
		gConfigFPGA.cntrlRegValue &= ~kMaskFpgaCCLK;
	else
		gConfigFPGA.cntrlRegValue |= kMaskFpgaCCLK;

	return
		SI_PLX_WritePCI_OpReg
		(
			gConfigFPGA.pPLXDev, 1, PLX_CNTRL << 2, &gConfigFPGA.cntrlRegValue
		);
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_WriteDATA
//
//	Description: Writes the data bit to FPGA, since FPGA has a serial interface,
//		data must be written as bits, bit by bit.
//
//	Parameters:
//		UINT32 bit:		Either an "1" or "0" to be written.
//


INT32 SI_ConfigFPGA_WriteDATA(UINT32 bit)
{
	if (bit == 0)
		gConfigFPGA.cntrlRegValue &= ~kMaskFpgaDIO;
	else
		gConfigFPGA.cntrlRegValue |= kMaskFpgaDIO;

	return
		SI_PLX_WritePCI_OpReg
		(
			gConfigFPGA.pPLXDev, 1, PLX_CNTRL << 2, &gConfigFPGA.cntrlRegValue
		);
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_WritePROG
//
//	Description: Writes the program bit to FPGA, this bit forces FPGA
//				into the Program mode.
//	Parameters:
//		UINT32 bit:		"1" = program mode; "0" = user mode.
//

INT32 SI_ConfigFPGA_WritePROG(UINT32 bit)
{
	if (bit == 0)
		gConfigFPGA.cntrlRegValue &= ~kMaskFpgaPROGRAM;
	else
		gConfigFPGA.cntrlRegValue |= kMaskFpgaPROGRAM;

	return
		SI_PLX_WritePCI_OpReg
		(
			gConfigFPGA.pPLXDev, 1, PLX_CNTRL << 2, &gConfigFPGA.cntrlRegValue
		);
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_ReadDONE
//
//	Description: Reads FPGA Done bit, this allows us to know when FPGA
//				has finished bit loading.
//	Parameters:
//		UINT32 *bit:	"1" = Finished Bit Loading; "0" = not ready yet.
//

INT32 SI_ConfigFPGA_ReadDONE(UINT32 *bit)
{
	INT32 error = 
		SI_PLX_ReadPCI_OpReg
		(
			gConfigFPGA.pPLXDev, 1, PLX_CNTRL << 2, &gConfigFPGA.cntrlRegValue
		);
	if (error != e_Err_NoError)
		return error;

	*bit = gConfigFPGA.cntrlRegValue & kMaskFpgaDONE;
	if (*bit)
		*bit = 1;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	Initialize parameters for loading bit files into FPGAs.
//	When there are more than one single FPGA, additional
//	FPGAs are daisy chain connected, and data from their
//	bitfiles are also concatenated to program them as a
//	single chain. ConcatMagic numbers are numbers added in
//	between data from one FPGA and the next. To obtain them,
//	compile the bit file, and using PROMGEN, create the
//	corresponding HEX file, then create a second HEX file
//	linking the the target bit file as secondary FPGA, with
//	any other one as primary, then compare and locate the
//	beginning of the first file inside the second one.
//	The 8 bytes before the data for the second FPGA are
//	the magic numbers for this particular FPGA.
//	You can also locate them easily since most files begin
//	with "ffffffffaa995566", this will appear at the begining
//	of your HEX file, and appear repeatedly as many more
//	FPGA are daisy chained. Note that even this method is easy,
//	is not a guarantee to locate the beginning of each file,
//	since the data itself also can have this sequence.
//

INT32 SI_ConfigFPGA_Initialize(struct PLXDevice *pPLXDev)
{
	UINT32 cnt;

	// init all availble FPGA parameters
	cnt = 0;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XCS05XL";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x1aa2;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= 0x06;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0xa9;
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC2S50E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x133a4;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0x4c;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0xeb;
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC2S100E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x1a5cc;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= 0x69;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= 0x75;
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC2S200E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x2C01C;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0xb0;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= 0x09;
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC2S300E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x393D8;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0xe4;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0xf8;
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC3S100E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x11BDC;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x00;			// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0x46;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0xF9;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC3S250E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x29500;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x00;			// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0xA5;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0x42;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC3S500E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x45480;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x01;			// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0x15;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0x22;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC3S1200E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0x75394;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x01;			// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0xD4;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0xE7;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.fpgaTypes[cnt].typeString	= "XC3S1600E";
	gConfigFPGA.fpgaTypes[cnt].length		= 0xB62E4;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[0]= 0x30;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[1]= 0x01;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[2]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[3]= 0x00;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[4]= 0x50;
	gConfigFPGA.fpgaTypes[cnt].concatMagic[5]= 0x02;			// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[6]= (UINT8)0xD8;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagic[7]= (UINT8)0xBB;		// Assumed to be.
	gConfigFPGA.fpgaTypes[cnt].concatMagicCount= 8;
	cnt++;

	gConfigFPGA.totalFPGAAvailble = cnt;

	gConfigFPGA.bufferSize = kMaxBitFileSize;
	gConfigFPGA.buffer = (char *)malloc(gConfigFPGA.bufferSize);
	if (gConfigFPGA.buffer == NULL)
	{
		gConfigFPGA.bufferSize = 0;
		OS_Popup_MessageBox
		(
			"Error malloc.\n",
			"Loading FPGA",
			0
		);

		return e_Err_WindowsMemError;
	}

	gConfigFPGA.pPLXDev = pPLXDev;

	return 
		SI_PLX_ReadPCI_OpReg
		(
			gConfigFPGA.pPLXDev, 1, PLX_CNTRL << 2, &gConfigFPGA.cntrlRegValue
		);
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_LoadBitfileToMemory
//
//	Description: Loads bit files into memory.
//	Parameters:
//		UINT32 fpgaPosition:	Relative position of the given FPGA within
//								the Daisy Chain, if any.
//		char *fpgaPartNum			Name of the FPGA.
//		char *filename			Complete path to the dorresponding bitfile.
//

INT32 SI_ConfigFPGA_LoadBitfileToMemory
(
	UINT32 fpgaPosition, char *fpgaPartNum, char *filename
)
{
	INT32 error;
	UINT32 cnt, count;
	char message[1024];

	// parse the bit file, so we can compare; leave some room in beginning
	error = 
		SI_ConfigFPGA_ParseBitFile
		(
			filename, fpgaPartNum,
			gConfigFPGA.buffer + kConfigFPGA_ConcatMagicCountMax, 
			gConfigFPGA.bufferSize - kConfigFPGA_ConcatMagicCountMax, 
			&gConfigFPGA.byteStartPosition, &gConfigFPGA.byteCount
		);
	if (error != e_Err_NoError)
	{
		OS_Popup_MessageBox
		(
			"\n\nCould not parse the bit file.\n",
			"Loading FPGA",
			0
		);

		return error;
	}
	gConfigFPGA.byteStartPosition += kConfigFPGA_ConcatMagicCountMax;


	// see if the FPGA type matches any we have specified.
	for (cnt=0; cnt<gConfigFPGA.totalFPGAAvailble; cnt++)
	{
		if ( !strcasecmp(gConfigFPGA.fpgaTypes[cnt].typeString, fpgaPartNum) )
		{
			if ( gConfigFPGA.byteCount == gConfigFPGA.fpgaTypes[cnt].length )
			{
				// match found.
				gConfigFPGA.fpgaTypeIndex = cnt;

				// first FPGA needs no more
				if (fpgaPosition == 0)
					return e_Err_NoError;	

				count = gConfigFPGA.fpgaTypes[cnt].concatMagicCount;

				// not the first FPGA. Add the 8 byte magic numbers
				// at the beginning.
				gConfigFPGA.byteStartPosition -= count;
				gConfigFPGA.byteCount += count;

				for (cnt=0; cnt<count; cnt++)
					gConfigFPGA.buffer[gConfigFPGA.byteStartPosition+ cnt] = 
						gConfigFPGA.
							fpgaTypes[gConfigFPGA.fpgaTypeIndex].
								concatMagic[cnt];

				return e_Err_NoError;
			}

			// count is wrong for the FPGA name.
			sprintf
			(
				message,
				"\n\nFor %s, bit file size must be %d.\n", 
				gConfigFPGA.fpgaTypes[cnt].typeString, 
				gConfigFPGA.fpgaTypes[cnt].length
			);

			OS_Popup_MessageBox
			(
				message,
				"Loading FPGA",
				0
			);

			return e_Err_FPGA_TypeNotFound;
		}
	}

	// last item didn't match.
	sprintf
	(
		message,
		"\n\nCould not find the FPGA type of %s\n", 
		fpgaPartNum
	);

	OS_Popup_MessageBox
	(
		message,
		fpgaPartNum,
		0
	);

	return e_Err_FPGA_TypeNotFound;
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_ResetFPGA
//
//	Description:
//		Toggles program pin and other necessay stuff before actual bit loading
//	Parameters:		None.
//

INT32 SI_ConfigFPGA_ResetFPGA()
{
	INT32 error;
	UINT32 bit;

	// we need to toggle EESK from low to high to low to kickstart CNTRL reg
	// Make sure CCLK is low (EESK)
	error = SI_ConfigFPGA_WriteCCLK(0);
	if ( error != e_Err_NoError )
		return error;

	// Make sure CCLK is high (EESK)
	error = SI_ConfigFPGA_WriteCCLK(1);
	if ( error != e_Err_NoError )
		return error;

	// Make sure CCLK is low (EESK)
	error = SI_ConfigFPGA_WriteCCLK(0);
	if ( error != e_Err_NoError )
		return error;

	// PROGRAM pin low to clear the FPGA. (USERo)
	error = SI_ConfigFPGA_WritePROG(0);
	if ( error != e_Err_NoError )
		return error;

	SI_PLX_Sleep(10);	// wait 10 ms

	// PROGRAM pin high to begin the program cycle. (USERo)
	error = SI_ConfigFPGA_WritePROG(1);
	if ( error != e_Err_NoError )
		return error;

	SI_PLX_Sleep(10);	// wait 10 ms

	// see if DONE is low. if not, something is wrong so abort
	error = 
		SI_ConfigFPGA_ReadDONE(&bit);
	if ( error != e_Err_NoError )
		return error;

	if ( bit )
	{
		OS_Popup_MessageBox
		(
			"Done did not go low.\n",
			"Loading FPGA",
			0
		);

		return e_Err_FPGA_DoneNotLow;
	}

	return e_Err_NoError;
}	

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_Close
//
//	Description:	Close Configuration to the FPGA.
//	Parameters:		None.
//

INT32 SI_ConfigFPGA_Close()
{
	if (gConfigFPGA.buffer != NULL)
		free(gConfigFPGA.buffer);

	gConfigFPGA.bufferSize = 0;
	gConfigFPGA.buffer = NULL;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_Single
//
//	Description:
//		Configures one FPGA, but does not check done.
//
//	Parameters:
//		char *filename:			Pointer to bit file name.
//		char *fpgaPartNum:			Pointer to FPGA name.
//		void *messageWindow:	Pointer to message window to be used to	let
//								user know progress of the bit load process.
//

INT32 SI_ConfigFPGA_Single(char *filename, char *fpgaPartNum, void *messageWindow)
{
	UINT32 cnt, cnt1, bit, bitCount;
	INT32 error;
	char message[1024];

	sprintf(message, "\nLoading\n %s to\n %s\n", filename, fpgaPartNum);
	
	OS_Popup_SHOW_MESSAGE_ON_WINDOW
	(
		messageWindow,
		message
	);

	// begin writing to FPGA
	bitCount = 0;
	for 
	(
		cnt=gConfigFPGA.byteStartPosition; 
		cnt<(gConfigFPGA.byteStartPosition + gConfigFPGA.byteCount); 
		cnt++
	)
	{
		for (cnt1=0; cnt1<8; cnt1++)
		{
			bit = ( gConfigFPGA.buffer[cnt] >> ( 7 - cnt1 ) ) & 1;
			bitCount ++;

			//	Write to DIO
			error = SI_ConfigFPGA_WriteDATA(bit);
			if ( error != e_Err_NoError )
				return error;

			//	CCLK is high
			error = SI_ConfigFPGA_WriteCCLK(1);
			if ( error != e_Err_NoError )
				return error;

			//	CCLK is low
			error = SI_ConfigFPGA_WriteCCLK(0);
			if ( error != e_Err_NoError )
				return error;
		}

		if ( cnt % 1024 == 0 )	// update every 1k bytes
		{
			sprintf
			(
				message,
				"%d of %d loaded.\r", 
				cnt-gConfigFPGA.byteStartPosition, 
				gConfigFPGA.byteCount
			);

			OS_Popup_SHOW_MESSAGE_ON_WINDOW
			(
				messageWindow,
				message
			);
		}
	}

	sprintf
	(
		message,
		"%d of %d loaded.\n",
		gConfigFPGA.byteCount,
		gConfigFPGA.byteCount
	);

	OS_Popup_SHOW_MESSAGE_ON_WINDOW
	(
		messageWindow,
		message
	);

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	SI_ConfigFPGA_CheckDone
//
//	Description:
//		Check done
//
//	Parameters:
//		UINT32 *status: Pointer to Done status.
//

INT32 SI_ConfigFPGA_CheckDone(UINT32 *status)
{
	UINT32 cnt;
	INT32 error;

	// toggle CCLK some more to get correct setup
	error = SI_ConfigFPGA_WriteDATA(0);
	if ( error != e_Err_NoError )
		return error;

	for (cnt=0; cnt<1024; cnt++)
	{
		//	CCLK is high
		error = SI_ConfigFPGA_WriteCCLK(1);
		if ( error != e_Err_NoError )
			return error;

		//	CCLK is low
		error = SI_ConfigFPGA_WriteCCLK(0);
		if ( error != e_Err_NoError )
			return error;
	}

	// check the DONE bit
	error = 
		SI_ConfigFPGA_ReadDONE(status);
	if ( error != e_Err_NoError )
		return error;

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	LoadBoardFPGAs
//
//	Description:
//		Load all FPGAs in a single board.
//
//	Parameters:
//		UINT32 boardID:				Slot number of board.
//		PLXDevice *pPLXDev:			Handle to the PLX device (board).
//		char *descriptionFilePath:	Complete path to the description file.
//

int LoadBoardFPGAs
(
	UINT32 boardID, struct PLXDevice *pPLXDev, char *descriptionFilePath
)
{
	UINT32 cmdBufSize = 4096, numberOfFPGAs;
	char *cmdBuffer, *fpgaPartNums[kMaxFPGAs], *bitfilenames[kMaxFPGAs];
	int error;
	UINT32 status, cnt;
	char message[1024];
	void *messageWindow = NULL;


	// allocate buffer.
	cmdBuffer = (char*)malloc(cmdBufSize);
	if (cmdBuffer == NULL)
	{
		OS_Popup_MessageBox
		(
			"\nMemory allocation error.\n\n",
			"Loading FPGA",
			0
		);

		return e_Err_WindowsMemError;
	}

	error = 
		ParseDescriptionFile
		(
			pPLXDev, boardID, descriptionFilePath,
			cmdBufSize, cmdBuffer, &numberOfFPGAs, fpgaPartNums, bitfilenames
		);
	if (error != 0)
	{
		free(cmdBuffer);
		return error;
	}

	// init data structure
	error = SI_ConfigFPGA_Initialize(pPLXDev);
	if ( error != e_Err_NoError )
	{
		free(cmdBuffer);
		return error;
	}

	// check each bit file by trying to load them
	for (cnt = 0; cnt < numberOfFPGAs; cnt++)
	{
		error = 
			SI_ConfigFPGA_LoadBitfileToMemory
			(
				cnt, fpgaPartNums[cnt], bitfilenames[cnt]
			);
		if ( error != e_Err_NoError )
		{
			SI_ConfigFPGA_Close();
			free(cmdBuffer);
			return error;
		}
	}

	// setup for new bit file
	error = SI_ConfigFPGA_ResetFPGA();
	if ( error != e_Err_NoError )
	{
		SI_ConfigFPGA_Close();
		free(cmdBuffer);
		return error;
	}

	OS_Popup_CREATE_WINDOW
	(
		&messageWindow,
		"Loading FPGA"
	);

	// load from file and send to FPGA
	for (cnt = 0; cnt < numberOfFPGAs; cnt++)
	{
		// load to memory
		error = 
			SI_ConfigFPGA_LoadBitfileToMemory
			(
				cnt, fpgaPartNums[cnt], bitfilenames[cnt]
			);
		if ( error != e_Err_NoError )
		{
			SI_ConfigFPGA_Close();
			free(cmdBuffer);
			return error;
		}

		// download bit file
		error = SI_ConfigFPGA_Single(bitfilenames[cnt], fpgaPartNums[cnt], messageWindow);
		if ( error != e_Err_NoError )
		{
			if(messageWindow)
				OS_Popup_DESTROY_WINDOW(messageWindow);
			SI_ConfigFPGA_Close();
			free(cmdBuffer);
			return error;
		}

		// extra cclk and check for done
		error = 
			SI_ConfigFPGA_CheckDone(&status);
		if ( error != e_Err_NoError )
		{
			if(messageWindow)
				OS_Popup_DESTROY_WINDOW(messageWindow);
			SI_ConfigFPGA_Close();
			free(cmdBuffer);
			return error;
		}

		if ( status && (cnt < (numberOfFPGAs - 1)) )
		{
			sprintf(message, "\nError: FPGA Done line high before all FPGA are Loaded.\
				\nOnly %d FPGAs were loaded\
				\nPlease verify Daughter Card is installed properly\n", cnt + 1);

			OS_Popup_MessageBox
			(
				message,
				"Loading FPGA",
				0
			);
			if(messageWindow)
				OS_Popup_DESTROY_WINDOW(messageWindow);

			free(cmdBuffer);
			return e_Err_FPGA_NotAllFPGALoaded;
		}

	}

	if(messageWindow)
		OS_Popup_DESTROY_WINDOW(messageWindow);

	if ( !status )
	{
		OS_Popup_MessageBox
		(
			"\nError: FPGA Done line did not go high.\n",
			"Loading FPGA",
			0
		);

		free(cmdBuffer);
		return e_Err_FPGA_DoneNotHigh;
	}

	OS_Popup_MessageBox
	(
		"\nConfiguration successful.\n",
		"Loading FPGA",
		0
	);

	// finish off
	SI_ConfigFPGA_Close();

	free(cmdBuffer);
	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	ParseCommandLine
//
//	Description:
//		Parse Command Line.
//
//	Parameters:
//		int argc:			Count of arguments.
//		char *argv[]:		List of arguments.
//		UINT32 *boardID:	Slot number of board.
//		char **filepath:	Pointer to description file path.
//

int ParseCommandLine
(
	int argc, char *argv[], UINT32 *boardID, char **filepath
)
{
	char *caption, message[1024] = "";

	if (argc != 3)
	{
		caption = "Board ID and a filename with FPGA descriptions must be present.";

		strcat(message, "For example, \n");
		strcat(message, "  fpgaload 0 fpgafile.txt\n");
		strcat(message, "where fpgafile.txt contains the following.\n");
		strcat(message, "  XCS05XL\nC:\\sic30dsp\\bitfile.bit0\n");
		strcat(message, "  XC2S100E\nC:\\sic30dsp\\bitfile.bit1\n");
		strcat(message, "  XC2S200E\nC:\\sic30dsp\\bitfile.bit2\n");

		OS_Popup_MessageBox
		(
			message,
			caption,
			0
		);

		return e_Err_InvalidNumOfParams;
	}

	sscanf(argv[1], "%d", boardID);
	*filepath = argv[2];

	return e_Err_NoError;
}

///////////////////////////////////////////////////////////////////////////////
//	ParseDescriptionFile
//
//	Description:
//		Parse description file.
//
//	Parameters:
//		PLXDevice *pPLXDev:			Handle to board.
//		UINT32 boardID:				Slot number of board.
//		char *descriptionFilePath:	Complete path to description file.
//		int bufferSize:				Size of buffer.
//		char *buffer:				Buffer to hold contents of
//									description file.
//		UINT32 *numberOfFPGAs:		Number of FPGAs in the board.
//		char *fpgaPartNums[]:			List of FPGA names in the board.
//		char *bitfilenames[]:		List of paths to corresponding bit files
//									for the FPGAs in the board.
//

int ParseDescriptionFile
(
	struct PLXDevice *pPLXDev, UINT32 boardID, char *descriptionFilePath,
	int bufferSize, char *buffer, UINT32 *numberOfFPGAs,
	char *fpgaPartNums[], char *bitfilenames[]
)
{
	FILE *fp;
	int bytesRead, bufPos, error;
	UINT32 value, temp, cnt;
	char *stopstr;

	// open the description file
	fp = fopen( descriptionFilePath, "rt" );
	if (fp == NULL)
	{
		OS_Popup_MessageBox
		(
			"FPGA description file does not exist.\n",
			"Loading FPGA",
			0
		);

		return e_Err_FileNotFound;
	}

	bytesRead = fread( buffer, 1, bufferSize, fp );
	fclose(fp);

	// null terminate buffer
	buffer[bytesRead] = 0;

	// now parse the file
	bufPos = 0;
	*numberOfFPGAs = 0;

	// read DID and VID
	error =
		SI_PLX_ReadPCI_NVWord
	(
		 pPLXDev, 1, 0, &value
	);
	if (error != e_Err_NoError)
	{
		return error;
	}

	temp = 0;
	while(1)
	{
		if (buffer[bufPos] == '[')
		{
			bufPos += 4;
			temp = strtol(&buffer[bufPos], &stopstr, 16);
			for (cnt = 0; cnt < 8; cnt++)
				bufPos++;

			if (value == temp)
				break;
		}
		if (bufPos >= bytesRead)
		{
			OS_Popup_MessageBox
			(
				"FPGA description file does not support boardID.\n",
				"Loading FPGA",
				0
			);

			return e_Err_FPGA_TypeNotFound;
		}
		bufPos++;
	}

	temp = 0;
	while((*numberOfFPGAs) < kMaxFPGAs)
	{
		// strip out stuff until valid char such as '0' or 'a'
		do
		{
			if (bufPos >= bytesRead)
				break;	// done

			if ( (buffer[bufPos] > ' ') && (buffer[bufPos] <= '~') )
				break;
			bufPos++;
		}while(1);

		if (bufPos >= bytesRead)
			break;	// done
		if (buffer[bufPos] == '[')
			break;

		if(temp == 0)
		{
			fpgaPartNums[*numberOfFPGAs] = &buffer[bufPos];
			temp++;
		}
		else
		{
			bitfilenames[*numberOfFPGAs] = &buffer[bufPos];
			(*numberOfFPGAs)++;
			temp = 0;
		}

		// look for something other than delimiter
		do
		{
			if (bufPos >= bytesRead)
				break;	// done

			if ( (buffer[bufPos] <= ' ') || (buffer[bufPos] > '~') )
			{
				buffer[bufPos] = 0;	// null terminate string
				break;
			}
			bufPos++;
		}while(1);

		if (bufPos >= bytesRead)
			break;	// done
		if (buffer[bufPos] == '[')
			break;
	}

	if(temp == 1)
	{	
		OS_Popup_MessageBox
		(
			"FPGA without bit file",
			"Parsing FPGA Description File",
			0
		);

		return e_Err_InvalidNumOfParams;
	}

	if((*numberOfFPGAs) >= kMaxFPGAs)
	{	
		OS_Popup_MessageBox
		(
			"Exceeded Maximum Number of FPGAs",
			"Parsing FPGA Description File",
			0
		);

		return e_Err_FPGA_MaxNumberOfFPGAs;
	}

	return e_Err_NoError;
}








