///////////////////////////////////////////////////////////////////////////////
//	sisample.cpp	: this is the filename for SI Hardware sample Program.
//	sample.cpp		: this is the filename for SIDDK only sample program. 
//
//	Description:
//		Main entry point for the command line sample program.
//		To change this only for SIDDK use, change the following:
//			1. Duplicate sisample.cpp to sample.cpp and sisample.h to sample.h
//			2. Remove functions that are SI board specific by deleting
//				functions that are inside #SIDEF_PLX_C671x and #SIDEF_PLX_C3x
//				in sample.cpp, sample.h.
//			3. change compiler switch definition for SIDDK
//				a.	Select Project->Settings
//				b.	Change to Settings for "All Configurations"
//				c.	Click on the C/C++ tab.
//				d.	Append SIDEF_PLX_SIDDK at the end of preprocessor 
//						definitions.
//				e.	Remove SIDEF_PLX_C3x/SIDEF_PLX_C671x if any exists.
//				f.	Click Ok.
//
//	Revision History:
//		2002-06-11: Av&Ley
//			Created sample program for SIDDK (adapted from C6711 sample 
//			program)
//		2002-07-01: mik
//			Changed all instances of SIHW_ to SI_
//		2002-07-24: mik
//			Fixed opening driver error checking.
//		2003-06-04: mik
//			Made it work with DOS "driver".
//		2003-06-09: mik
//			Changed buffer size to 1000 to reduce stack usage.
//		2003-06-19: mik
//			Added Sample_DSPInitiated_Access.
//		2003-07-09: mik
//			Coffload/verify returns DRAM parameters.
//		2006-04-06: Whipple
//			Coffload active is now a 2 step load (passive and active)
//		2006-06-09: Whipple
//			Fixed to work for Linux again
//		2006-06-23: Whipple
//			Removed erroneous DSP memory map menu items
//			Fixed config space read for WinXP
//		2006-07-03: Whipple
//			Combined active and passive coffloads
//		2006-09-01: Whipple
//			Removed Rev0 Comm mode e_CommModeIdx_Addon_Async_IO_Int
//		2007-02-01: Madrid
//			Fixed Messaging for Linux C33
//		2007-07-13: Ley
//			Updated C33 for messaging, except for SetXferSize which
//			it doesn't use.
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
//#include <sys/mman.h>	// for debug

#include "sisample.h"
#ifdef MOD64
#include "../../../../../si_modules/mod64/mod64.h"
#endif
#ifdef MOD68
#include "../../../../../si_modules/mod68/mod68.h"
#endif

#include "../../../../../common_sihw/cboard.h"

// in order to support DOS, this has to be less than 16k DWords
// but we set it to 4k DWords to leave room for other stuff.
#define kMaxBufferSize	( 4096 )

///////////////////////////////////////////////////////////////////////////////

//PLXDevice gPLXDev, *pPLXDev = &gPLXDev;
//CPLXBoard *board;
struct PLXC33 board;

void callback(int param)
{
	INT32 error;
	UINT32 flag;
	static int trigger = 0;

	printf("\nReceived Signal %d Interrupt #%d generated\n", param,
          ++trigger);
	
	//check for message
	error = 
		SI_PLXC33_ReadTarget
		(
			board.cgpSIDev, 
			0, 1, kPLXC33_DSPMessageFlag, &flag
		);
	/*if ( error != e_Err_NoError )			//This function returns void
		return error;*/

	if(flag & kPLXC33_RecvMsgFlag)
	{
		board.cgpSIDev->eventMsg[0].msgType = EVENT_MSG_TYPE;
		board.cgpSIDev->eventMsg[0].msgText[0] = 1;
		if (msgsnd(board.cgpSIDev->msgID, &board.cgpSIDev->eventMsg[0], sizeof(char), 0) == -1)
		{
			printf("Error in msgsnd.\n");	
			return;
		}
		//board->cgpSIDev->message_event = 1;
	}
	else
		board.cgpSIDev->wait_event = 1;
}
///////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
	INT32 error;
	UINT32 boardID;

	// Display the Banner
	printf("\nSI-DDK PLX Sample Application 4.0\n");
	printf("Sheldon Instruments, Inc. 2007\n");
	printf("Supporting the PLX 90xx PCI chip\n");

	// if a device selection is made on the command line, use it
	if (argc > 1) 
		//convert from ASCII to numeric
		sscanf(argv[1], "%d", &boardID);
	else 
		boardID = 0;	

	printf("Using Device Number %d\n\n", boardID);

	printf("NOTE: Remember to use option F to load FPGA\n");
	printf("before continuing with the rest of the functions\n\n");

//	board = new CPLXC33();
//	if(board == NULL) return e_Err_DeviceNotFound;
	error = ClassLoad( &boardID );

	if (error != e_Err_NoError)
	{
		printf("Device Not Found. Press enter to continue. Code=%d.\n", error);
		getchar();
		return error;
	}

	// check for addon init buffer
	if (board.cgpSIDev->addonInitBufferSize == 0)
		printf("Add On Initiated Buffer not found.\n");


	error = SI_PLX_SetupCallbackEvent( board.cgpSIDev, (void *)callback );
	if (error != e_Err_NoError)
	{
		printf
		(
			"Setup callback failed. Press enter to continue. "
			"Code=%d.\n", error
		);
		getchar();
		return error;
	}

	error = SI_PLX_SetupMessageEvent( board.cgpSIDev );
	if (error != e_Err_NoError)
	{
		printf
		(
			"Setup messaging failed. Press enter to continue. "
			"Code=%d.\n", error
		);
		getchar();
		return error;
	}

	printf("\nPress Enter to Continue.\n");
	getchar();

	// 4. We have a handle to the driver. Go do sample stuff.
	Sample_MainCommandLoop();

	// 5. Sample stuff ended (error or otherwise). Now close the driver
	error = ClassUnload();
//	board = NULL;
	
	return e_Err_NoError;
}


///////////////////////////////////////////////////////////////////////////////

void Sample_MainCommandLoop()
{
	bool Done = false;
	char user_input[2];

	UINT32 *dirAccessAddr[6];

	// Get Parameters for direct access
	if 
	(
		SI_PLX_DirectAccessAddr
		(
			board.cgpSIDev,  6, dirAccessAddr
		) != e_Err_NoError
	)
	{
		printf
		(
			"Driver reported an error in Get Baddr. "
			"Direct access will not work.\n"
		);
	}
	
	// Enter the user command loop
	while(!Done)
	{
		printf(kMENU);
		scanf("%1s", user_input);
		getchar();

		// Now call the Driver as necessary to perform useful work
		switch(user_input[0])
		{		
		case '1':
			Sample_WriteMailboxRegister();
			break;

		case '2':
			Sample_ReadMailboxRegister();
			break;

		case '3':
			Sample_ReadConfigurationSpace();
			break;

		case '4':
			Sample_WriteOpReg();
			break;

		case '5':
			Sample_ReadOpReg();
			break;

		case '6':
			Sample_WriteBusmastered();
			break;
		
		case '7':
			Sample_ReadBusmastered();
			break;

		case '8':
			Sample_WriteTarget();
			break;

		case '9':
			Sample_ReadTarget();
			break;

		case 'a':
		case 'A':
			Sample_WriteNVRAM();
			break;

		case 'b':
		case 'B':
			Sample_ReadNVRAM();
			break;

		case 'c':
		case 'C':
			Sample_SetTimeout();
			break;

		case 'd':
		case 'D':
			Sample_DirectAccess(dirAccessAddr);
			break;

		case 'e':
		case 'E':
			Sample_DriverConfigPointBlock();
			break;

#ifdef SIDEF_PLX_C3xC671x
		case 'F':
		case 'f':
			Sample_ConfigureFPGA();
			break;

		case 'G':
		case 'g':
			Sample_LoadCofffile();
			break;

		case 'H':
		case 'h':
			Sample_DSPComm();
			break;
#ifdef MOD64
		case 'M':
		case 'm':
			Sample_6400(board.cgpSIDev, board.boardInfo.dspBusClkSpeed);
			break;
#endif

#ifdef MOD68
		case 'M':
		case 'm':
			Sample_6800(board.cgpSIDev, board.boardInfo.dspBusClkSpeed);
			break;
#endif

#endif
		case '0':
			printf("\nApplication Exiting......\n");
			Done = true;
			break;

		default:
			printf("\nInvalid Option.");
			break;
		}

		printf("\nPress Enter to Continue.\n");
		getchar();
	}
}

///////////////////////////////////////////////////////////////////////////////

void Sample_WriteMailboxRegister()
{
	INT32 error;
	UINT32 mailboxNum, value, *data;

	// mailbox write
	// choose the mailbox and it's desired value
	printf("\nChoose an outgoing register for writing (0-7) :");
	scanf("%d",&mailboxNum);
	getchar();
	printf("\nData to write: 0x");
	scanf("%x", &value);
	getchar();

	data = &value;

	error = 
		SI_PLX_WriteMailbox
		(
			board.cgpSIDev,  
			1, mailboxNum, data
		);
	if (error != e_Err_NoError)
	{
		printf("Driver reported an error in Mailbox Write. Code=%d\n", error);
		return;
	}

	printf("\nMailbox data written successfully\n");
}

///////////////////////////////////////////////////////////////////////////////

void Sample_ReadMailboxRegister()
{
	INT32 error;
	UINT32 mailboxNum, value, *data;

	// mailbox read
	// choose the mailbox
	printf("\nChoose a mailbox register for reading (0-7): ");
	scanf("%d", &mailboxNum);
	getchar();

	data = &value;

	error = SI_PLX_ReadMailbox
		(
			board.cgpSIDev,  
			1, mailboxNum, data
		);
	if (error != e_Err_NoError)
	{
		printf("Driver reported an error in Mailbox Read. Code=%d\n", error);
		return;
	}

	printf("\n0x%X is in the mailbox\n", value);
}

///////////////////////////////////////////////////////////////////////////////

void Sample_ReadConfigurationSpace()
{
	INT32 error;
	UINT32 cnt, cnt1;
	UINT32 bufferUINT32[kMaxBufferSize];		// general purpose   0x40 / 4
	unsigned char *bufferUINT8 = (unsigned char *)(bufferUINT32);

	// read PCI configuration space
	error = 
		SI_PLX_ReadPCI_ConfSpace
		(
			board.cgpSIDev,  
			0x54, 0, bufferUINT32
		);
	if (error != e_Err_NoError)
	{
		printf("Driver couldn't read configuration space. Code=%d.\n", error);
		return;
	}

	for(cnt=0; cnt<0x54; cnt+=4)
	{
		printf(" 0x%2.2X ", cnt);
		for(cnt1=0; cnt1<4; cnt1++)
			printf("%2.2x ", bufferUINT8[cnt+(3-cnt1)]);
		printf("\n");
	}
}

///////////////////////////////////////////////////////////////////////////////

void Sample_WriteOpReg()
{
	INT32 error;
	UINT32 opRegNum, value, *data;

	// OpReg Write
	printf("\nEnter OpReg Offset (Byte Boundary) to Write: 0x");
	scanf("%x", &opRegNum);
	getchar();

	printf("\nEnter Data to Write to OpReg: 0x");
	scanf("%x", &value);
	getchar();

	data = &value;

	error = 
		SI_PLX_WritePCI_OpReg
		(
			board.cgpSIDev,  
			1, opRegNum, data
		);
	if (error != e_Err_NoError)
	{
		printf("Driver reported an error in OpReg Write. Code=%d.\n", error);
		return;
	}

	printf
	(
		"\n0x%X (%d dec) written to OpReg\n", opRegNum, value
	);
}

///////////////////////////////////////////////////////////////////////////////

void Sample_ReadOpReg()
{
	INT32 error;
	UINT32 opRegNum, value, *data;

	// OpReg Read
	printf("\nEnter OpReg Offset (Byte Boundary) to Read: 0x");
	scanf("%x", &opRegNum);
	getchar();

	data = &value;

	error = 
		SI_PLX_ReadPCI_OpReg
		(
			board.cgpSIDev,  
			1, opRegNum, data
		);
	if (error != e_Err_NoError)
	{
		printf("Driver reported an error in OpReg Read. Code=%d.\n", error);
		return;
	}

	printf("\n0x%X read from OpReg\n", value);
}



///////////////////////////////////////////////////////////////////////////////

// Because of a problem with how Windows handles DeviceIOControl calls
// for Busmastered Reads and Writes, for these two calls only,
// the API is different. For these two, the parameters are passed as
// parameters, sizeof(parameters), data, sizeof(data), rather than the
// more common "in, sizeof(in), out, sizeof(out)" ordering.

void Sample_WriteBusmastered()
{
	INT32 error;
	UINT32 offset, count;
	UINT32 cnt;
	UINT32 ramp[kMaxBufferSize];

	offset = 0x0; //intialize offset to zero.

	PrintAvailableAddresss(kGetAddress_HostView);
	
	GetAddonAddress(&offset);
	printf("\nEnter the DWORD Count (1 - %d): ", kMaxBufferSize);
	scanf("%d", &count);
	getchar();

	// create a ramp pattern to dump to the FIFO
	for(cnt=0; cnt<count; cnt++)
		ramp[cnt] = cnt;

	printf("\nAsync Write in Progress\n");

	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	// it is 24bit DWORD offset Address as seen by C33 DSP
	error = 
		SI_PLX_WriteBlockDMA
		(
			board.cgpSIDev,  
			count, offset, ramp 
		);

	if (error != e_Err_NoError)
	{
		printf
		(
			"Driver reported an error in Busmastered Write. "
			"Code=%d.\n", error
		);
		return;
	}

	printf("Successful Transfer, Data Written");

	return;
}

///////////////////////////////////////////////////////////////////////////////

// Because of a problem with how Windows handles DeviceIOControl calls
// for Busmastered Reads and Writes, for these two calls only,
// the API is different. For these two, the parameters are passed as
// parameters, sizeof(parameters), data, sizeof(data), rather than the
// more common "in, sizeof(in), out, sizeof(out)" ordering.

void Sample_ReadBusmastered()
{
	INT32 error;
	UINT32 offset, count;
	UINT32 cnt;
	UINT32 ramp[kMaxBufferSize];
	UINT32 t=0;
	
	offset = 0x0; //intialize offset to zero.
//	mprotect(ramp, kMaxBufferSize, PROT_WRITE);	// debug

	for(cnt = 0; cnt < kMaxBufferSize; cnt++)	// debug
		ramp[cnt]=0;					// debug

	PrintAvailableAddresss(kGetAddress_HostView);
	
	GetAddonAddress(&offset);
	printf("\nEnter the DWORD Count (1 - %d): ", kMaxBufferSize);
	scanf("%d",&count);
	getchar();

	printf("\nAsync Read in Progress\n");

	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	error = 
		SI_PLX_ReadBlockDMA
		(
			board.cgpSIDev,  
			count, offset, ramp
		);

	if (error != e_Err_NoError)
	{
		printf
		(
			"Driver reported an error in Busmastered Read. "
			"Code=%d.\n", error
		);
		return;
	}

	printf("Successful Transfer, Data Read:\n");

// debug

	for(cnt=0;cnt<count;cnt++)
	{
		printf("%32x\t0x%08x\n", offset + cnt * board.boardInfo.cgDSP_DataSize, ramp[cnt]);

		if( (cnt+1) % 16 == 0)
			printf("\n");
	}


// debug
	for(cnt=0; cnt<count; cnt++)
	{
		if(ramp[cnt] == cnt)
			++t;
	}
	printf("Found %d ramp values\n", t);	
	return;
}

///////////////////////////////////////////////////////////////////////////////

void Sample_WriteTarget()
{

	INT32 error;
	UINT32 region, offset, count, cnt;

	UINT32 bufferUINT32[kMaxBufferSize];		// general purpose
	
	offset = 0x0 ; //Intialize offset to zero.
	
	// Target write
	printf("\nEnter write region (Space 0 or 1): ");
	scanf("%x",&region);
	getchar();

	PrintAvailableAddresss(kGetAddress_HostView);
	GetAddonAddress(&offset);

	printf("\nEnter Count (1 - %d): ", kMaxBufferSize);
	scanf("%d",&count);
	getchar();

	for(cnt=0; cnt<count; cnt++)
	{
		printf ("\nEnter value %d for writing: 0x", cnt);
		scanf("%x", &(bufferUINT32[cnt]));	// 32-bit
		getchar();
	}
	
	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	error = 
		SI_PLX_WriteTarget
		(
			board.cgpSIDev,  region, 
			count, offset, bufferUINT32			
		);

	if (error != e_Err_NoError)
	{
		printf ("\nWRITE FAILED! Code=%d.\n", error);
		return;
	}

	printf ("\nWrite Successful!\n");

}

///////////////////////////////////////////////////////////////////////////////

void Sample_ReadTarget()
{
	INT32 error;
	UINT32 region, offset, count, cnt;
	
	UINT32 bufferUINT32[kMaxBufferSize];		// general purpose

	offset = 0x0 ; //Intialize offset to zero.

	// Target read
	printf("\nEnter read region (Space 0 or 1): ");
	scanf("%x", &region);
	getchar();

	PrintAvailableAddresss(kGetAddress_HostView);
	GetAddonAddress(&offset);

	printf("\nEnter Count (1 - %d): ", kMaxBufferSize);
	scanf("%d", &count);
	getchar();

	// offset represents the dsp Address on the C33
	// it is not byte boundary offset expected by the API
	error = 
		SI_PLX_ReadTarget
		(
			board.cgpSIDev,  region, 
			count, offset, bufferUINT32
		);

	printf("\nOffset Address (DWORD Boundary) :   Value Read\n");

	if (error != e_Err_NoError)
	{
		printf("\nREAD FAILED. Code=%d.\n", error);
		return;
	}

	for(cnt=0; cnt<count; cnt++)	// 32-bit
		printf("%32x\t0x%08x\n", offset + cnt * board.boardInfo.cgDSP_DataSize, bufferUINT32[cnt]);

	printf ("\nRead Successful!\n");


}

///////////////////////////////////////////////////////////////////////////////

void Sample_WriteNVRAM()
{
	INT32 error;
	UINT32 addrNVRam, value, *data;

	// query the user for NV Write

	printf("\nChoose write offset (Byte Boundary): 0x");
	scanf("%x", &addrNVRam);
	getchar();
	printf("\nData to write: 0x");
	scanf("%x", &value);
	getchar();

	data = &value;

	error = 
		SI_PLX_WritePCI_NVWord
		(
			board.cgpSIDev,  
			1, addrNVRam, data
		);
	if (error != e_Err_NoError)
	{
		printf("\nWrite Failed. Code=%d.\n", error);
		return;
	}

	printf("\nNvRam data written successfully\n");
}

///////////////////////////////////////////////////////////////////////////////

void Sample_ReadNVRAM()
{
	INT32 error;
	UINT32 addrNVRam, value, *data;

	// NVRAM read

	printf("\nChoose Read offset (Byte Boundary): 0x");
	scanf("%x", &addrNVRam);
	getchar();

	data = &value;

	error = 
		SI_PLX_ReadPCI_NVWord
		(
			board.cgpSIDev,  
			1, addrNVRam, data
		);
	if (error != e_Err_NoError)
	{
		printf("Driver reported an error in NvRam Read. Code=%d.\n", error);
		return;
	}

	printf("\n0x%X is read from nvram %X", value, addrNVRam);
}

///////////////////////////////////////////////////////////////////////////////

void Sample_SetTimeout()
{
	INT32 error;
	UINT32 count;

	// Set Busmaster timeout
	printf("\nEnter new timeout (ms...0 = No timeout): ");
	scanf("%d", &count);
	getchar();

	error = 
		SI_PLX_SetTimeout
		(
			board.cgpSIDev,  count
		);

	if (error != e_Err_NoError)
	{
		printf("Driver reported an error in Set Timeout. Code=%d.\n", error);
		return;
	}
}

///////////////////////////////////////////////////////////////////////////////

void Sample_DirectAccess(UINT32 *dirAccessAddr[])
{
	UINT32 cnt, cnt1, readWrite, region, offset, count;
	UINT32 bufferUINT32[kMaxBufferSize];		// general purpose
	//UINT16 *bufferUINT16 = (UINT16 *)(bufferUINT32);
	//UINT8 *bufferUINT8 = (UINT8 *)(bufferUINT32);

	printf("\nValid Addresses: \n");
	for(cnt=0; cnt<6; cnt++)
		printf("\t%d 0x%x \n", cnt, (UINT32)dirAccessAddr[cnt]);

	printf("\nRead(0) or Write(1): ");
	scanf("%d", &readWrite);
	getchar();
	printf("\nRegion: ");
	scanf("%d", &region);
	getchar();
	printf("\nOffset (BYTE Boundary): 0x");
	scanf("%x", &offset);
	getchar();
	printf("\nEnter the Count (1 - %d): ", kMaxBufferSize);
	scanf("%d", &count);
	getchar();

	// Write
	if(readWrite)
	{
		for(cnt=0; cnt<count; cnt++)
		{
			printf("\nEnter value %d for writing: 0x", cnt);
			scanf("%x", &(bufferUINT32[cnt]));
			getchar();
		}

		cnt = 0;

		for(cnt1=offset; cnt1<offset+count;)
			dirAccessAddr[region][cnt1++] = bufferUINT32[cnt++];
	}
	// Read
	else
	{
		cnt = 0;
		
		for(cnt1=offset; cnt1<offset+count;)
			bufferUINT32[cnt++] = dirAccessAddr[region][cnt1++];
			
		for(cnt=0; cnt<count; cnt++)
			printf("%10x     %10x\n", offset+cnt, bufferUINT32[cnt]);

	}
}

///////////////////////////////////////////////////////////////////////////////

void Sample_DriverConfigPointBlock()
{
	INT32 error;
	UINT32 value;

	printf
	(
		"%s%d%s%d%s", 
		"Enter ", 
		kFlagDriverConfig_Block, " for block, ",
		kFlagDriverConfig_Point, " for point: "
	);
	scanf("%d", &value);
	getchar();

	//	Make sure it is block to block mode in beginning.
	error = 
		SI_PLX_DriverConfig_BlockPoint
		(
			board.cgpSIDev,  value
		);

	if ( error != e_Err_NoError )
		printf
		(
			"\nError Configuring Driver. Check Driver Version. "
			"Code=%d.\n", error
		);
}

///////////////////////////////////////////////////////////////////////////////

void GetAddonAddress(UINT32 *offset)
{
	printf("\nEnter Add-on/Local Address (DWORD Boundary): 0x");
	scanf("%x",offset);
	getchar();	
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Following code is only for SI boards

void CoffNormal()
{
	char filename[255];
	INT32 error;
	UINT32 dramSize1, dramSize2, totalMemory;

	// load coff file for C33
	UINT32 	*beforeCoff, beforeCoffCount;	
	UINT32 memoryParams[2];			// Use array to match C6711 SPD data format
	UINT32 	dspMemCount = kCOFFImage_MaxBufferSize / sizeof(UINT32),
			*passiveCoffImage, passiveCoffImageCount,
			*activeCoffImage, activeCoffImageCount, cnt;

	printf("\nEnter the full path for the COFF file:\n");
	printf("(example = c33ini.out for SI-DSP, c33inimc.out for SIDSP+SIMOD):\n");
	scanf("%s", filename);
	getchar();

	printf ("\nLoading COFF file \"%s\"\n",filename);
	
	beforeCoff = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
	if (beforeCoff == NULL)
		return;

	error = SI_PLXC33_CoffLoad_GetMemoryParams(board.cgpSIDev, memoryParams);
	if(error) return;

	board.pBoardInfo->cgDSP_Bank0 = memoryParams[0];
	board.pBoardInfo->cgDSP_Bank1 = memoryParams[1];

	// parameters to write before COFF file is written
	beforeCoffCount = 1;	// reserve first for count

	// if something must be added before Coff file, add it here and
	// increment beforCoffCount++ per each item, Follow following example
	// beforeCoff[beforeCoffCount++] = 0;			//type

	// actual number of words is one less than total (first is count)
	beforeCoffCount --;
	beforeCoff[0] = beforeCoffCount;

	passiveCoffImage = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
	if (passiveCoffImage == NULL)
	{
		free(beforeCoff);
		return;
	}

	activeCoffImage = (UINT32 *)malloc(kCOFFImage_MaxBufferSize);
	if (activeCoffImage == NULL)
	{
		free(beforeCoff);
		free(passiveCoffImage);
		return;
	}

	for(cnt = 0; cnt < dspMemCount; cnt++)
	{
		passiveCoffImage[cnt] = 0;
		activeCoffImage[cnt] = 0;
	}

	error = SI_PLXC33_ParseCoff(	
		board.cgpSIDev, filename, dspMemCount,
		passiveCoffImage, &passiveCoffImageCount,
		activeCoffImage, &activeCoffImageCount,
		beforeCoff);
	if(error != e_Err_NoError)
	{
//		printf("Error Parsing Coff File. Code =%d.\n", error);
		OS_Popup_ErrorMessageBox(
			"Error Parsing Coff File.", "Coff Parsing",
			error, 1 );
		free(beforeCoff);
		free(passiveCoffImage);
		free(activeCoffImage);
		return;
	}

// Start Passive Coff Load
	error = 
		SI_PLXC33_CoffWrite
		(
			board.cgpSIDev, filename, 0,//coffOptions, 
			passiveCoffImage, passiveCoffImageCount
		);
	if ( error != e_Err_NoError )
	{
//		printf("Error Loading Coff File. Code=%d.\n", error);
		OS_Popup_ErrorMessageBox(
			"Error Loading Coff File.", "Coff Write Passive",
			error, 1 );
		free(beforeCoff);
		free(passiveCoffImage);
		free(activeCoffImage);
		return;
	}

	// verify coff file
	error = 
		SI_PLXC33_CoffVerify
		(
			board.cgpSIDev, filename, 0,//coffOptions, 
			passiveCoffImage, passiveCoffImageCount
		);
	if ( error != e_Err_NoError )
	{
//		printf("Error Verifying Coff File. Code=%d.\n", error);
		OS_Popup_ErrorMessageBox(
			"Error Verifying Coff File.", "Coff Verify Passive",
			error, 1 );
		free(beforeCoff);
		free(passiveCoffImage);
		free(activeCoffImage);
		return;
	}
// End Passive Coff Load

	// COFF loaded and verified. Now run DSP by toggling reset
	error = 
		SI_PLXC33_ResetBoard
		(
			board.cgpSIDev,  e_Reset_Toggle
		);
	if ( error != e_Err_NoError )
	{
//		printf
//		(
//			"Error Toggling reset. DSP may not be running. "
//			"Code=%d.\n", error
//		);
		OS_Popup_ErrorMessageBox(
			"Error Toggling reset. DSP may not be running.",
			"Error During Reset",
			error, 1 );
		free(beforeCoff);
		free(passiveCoffImage);
		free(activeCoffImage);
		return;
	}

// Start Active Coff Load
	error = 
		SI_PLXC33_CoffWrite
		(
			board.cgpSIDev, filename, 1,//coffOptions, 
			activeCoffImage, activeCoffImageCount
		);
	if ( error != e_Err_NoError )
	{
//		printf("Error Loading Coff File. Code=%d.\n", error);
		OS_Popup_ErrorMessageBox(
			"Error Loading Coff File.", "Coff Write Active",
			error, 1 );
		free(beforeCoff);
		free(passiveCoffImage);
		free(activeCoffImage);
		return;
	}

	// verify coff file
	error = 
		SI_PLXC33_CoffVerify
		(
			board.cgpSIDev, filename, 1,//coffOptions, 
			activeCoffImage, activeCoffImageCount
		);
	if ( error != e_Err_NoError )
	{
//		printf("Error Verifying Coff File. Code=%d.\n", error);
		OS_Popup_ErrorMessageBox(
			"Error Verifying Coff File.", "Coff Verify Active",
			error, 1 );
		free(beforeCoff);
		free(passiveCoffImage);
		free(activeCoffImage);
		return;
	}
// End Active Coff Load

	dramSize1 = memoryParams[0] / (1 << 10);
	dramSize2 = memoryParams[1] / (1 << 10);
	totalMemory = dramSize1 + dramSize2;

	printf
	(
		"  Memory Size: %d KB \n"
		"  Bank 0 (CE0): %d KB \n  Bank 1 (CE1): %d KB \n",

		totalMemory,
		dramSize1,
		dramSize2
	);

	return;
}


void CoffPlusQuList()
{
	char filename[255];
	INT32 error;

	printf("\nEnter the full path for the COFF+QuList file:\n");
	scanf("%s", filename);
	getchar();

	printf("\nLoading COFF file \"%s\"\n",filename);

	error = CoffQuListfromFile(filename);
	if(error)
	{
		printf("Error loading COFF+QuList file\n");
		return;
	}

	error = ResetBoard( e_Reset_Toggle );
	if (error)
	{
		printf("Error reseting board\n");
		return;
	}

	printf("\nSuccessfully loaded COFF+QuList file\n");

	return;

}

///////////////////////////////////////////////////////////////////////////////

#ifdef SIDEF_PLX_C3xC671x

void Sample_LoadCofffile()
{
	char option[255];
	printf("\n 1  DSP COFF file (*.out) generated with TI tools.\n");
	printf(" 2  COFF+QuList file generated with SI QuX tools\n");
	scanf("%s", option);
	getchar();

	if(option[0] != '2')	// option[0] != '2' means this option is default
	{
		CoffNormal();
	}
	else
	{
		CoffPlusQuList();
	}
}


///////////////////////////////////////////////////////////////////////////////

void Sample_DSPComm()
{
	INT32 error;
	UINT32 mode, offset, count, readWrite, cnt, 
		buffer[1024];

	char tempString[1024];

	printf("\n");
	printf("0 = Data Transfers\n");
	printf("1 = Messaging\n");
	scanf("%d",&mode);
	getchar();

	if(mode)
	{
		printf("Enter number of messages to be exchanged (1 - 1000):");
		scanf("%d",&count);
		getchar();

		printf("\n");
		//set limits on ping pong

		if(count < 1)
			count = 1;
		else if(count > 1000)
			count = 1000;

		cnt = 0;
		while(cnt != count)
		{
			//printf("Sending Message to target, Message Recv'd: 0x%x\n", cnt);
			error = SI_PLXC33_SendMessage
			(
				// for demonstration purposes
				// use negative numbers to use Test Message Routine.
				board.cgpSIDev, -1, 0x80000000 + (++cnt)
			);
			if(error != e_Err_NoError)
			{
				printf
				(
					"Error sending DSP Message. "
					"Code=%d.\n", error
				);
				return;
			}
			//printf("Message sent: 0x%x, receiving message\n", count);

			error = SI_PLX_WaitMessageFromDSP(board.cgpSIDev, -1);
			if ( error != e_Err_NoError )
			{
				printf
				(
					"Error Receiving Message\n"
					"Code=%d.\n", error
				);
				return;
			}

			error = SI_PLXC33_RecvMessage(board.cgpSIDev, 0, &cnt);
			if(error != e_Err_NoError)
			{
				printf
				(
					"Error Receiving Message\n"
					"Code=%d.\n", error
				);
				return;
			}
		}

		return;
	}

	printf("\n");
	printf("Following modes are available:\n");
	for (cnt = 0; 1; cnt++)
	{
		error = 
			SI_PLXC33_TranslateCommModeIndex
			(
				cnt, &mode, 1024, tempString
			);
		if (error != e_Err_NoError)
			break;

		printf("    %d = %s\n", cnt, tempString);
	}
	printf("\n");
	scanf("%d",&mode);
	getchar();

	mode = 0;
	if((mode < 0) || (mode > e_CommModes_PLXC33_Point_Sync_IO_Target_CommReg))
	{
		printf("invalid mode\n");
		return;
	}

	printf("    0 = Read From DSP\n");
	printf("    1 = Write To DSP\n");
	scanf("%d", &readWrite);
	getchar();

	PrintAvailableAddresss(kGetAddress_DSPView);
	printf("Enter DSP Address: 0x");
	scanf("%x",&offset);
	getchar();
	
	printf("count:");
	scanf("%d",&count);
	getchar();

	// if writing to DSP, get all the data to write. 
	if (readWrite == 1)	
	{
		for (cnt=0; cnt<count; cnt++)
		{
			printf ("\nEnter value %d for writing: 0x",cnt);
			scanf ("%x", &buffer[cnt]);
			getchar();

		}
	}

	if (readWrite == 1)	
	{
		switch (mode)
		{
		// These 2 are PLX communication modes, not DSP
		// but use DSP addresses
		case e_CommModes_PLXC33_Target:
			error = SI_PLXC33_WriteTarget
				(
					board.cgpSIDev, 
					0, count, offset, buffer
				);
			break;

		case e_CommModes_PLXC33_BMDMA:
			error = SI_PLXC33_WriteBlockDMA
				(
					board.cgpSIDev, 
					count, offset, buffer
				);
			break;

		// The rest are DSP active modes
		default:
			error = SI_PLXC33_WriteDSPActive
				(
					board.cgpSIDev,  
					mode, count, offset, buffer
				);
			break;
		}
		{
			if (error != e_Err_NoError)
			printf
			(
				"Error occurred during DSP communications. "
				"Code=%d.\n", error
			);
			return;
		}
	}
	// if reading from DSP
	else
	{
		switch (mode)
		{
		// These 2 are PLX communication modes, not DSP
		case e_CommModes_PLXC33_Target:
			error = SI_PLXC33_ReadTarget
				(
					board.cgpSIDev, 
					0, count, offset, buffer
				);
			break;

		case e_CommModes_PLXC33_BMDMA:
			error = SI_PLXC33_ReadBlockDMA
				(
					board.cgpSIDev, 
					count, offset, buffer
				);
			break;

		// The rest are DSP active modes
		default:
			error = SI_PLXC33_ReadDSPActive
				(
					board.cgpSIDev,  
					mode, count, offset, buffer
				);
			break;
		}
		if (error != e_Err_NoError)
		{
			printf
			(
				"Error occurred during DSP communications. "
				"Code=%d.\n", error
			);
			return;
		}

		for (cnt=0; cnt<count; cnt++)
		{
			printf("%32x\t0x%08x\n", offset + cnt * board.boardInfo.cgDSP_DataSize, buffer[cnt]);//32-bit
		}	// close for
	}

/* todo:
	// if writing to DSP, get all the data to write. 
	if (readWrite == 1)	
	{
		switch(mode)
		{
		case kFlagDSPActiveComm_DSPDMA_HostInt:
		case kFlagDSPActiveComm_DSPIO_HostInt:
			error = 
				SI_PLXC33_WriteAddonInit
				(
					board->cgpSIDev,  
					count, offset, buffer, mode, callbackEvent, 
					addonInitBufferSize, addonInitBuffer
				);
			break;

		case kFlagDSPActiveComm_DSPIO_HostPoll:
		case kFlagDSPActiveComm_DSPDMA_HostPoll:
			error = 
				SI_PLXC33_WriteHostpoll
				(
					board->cgpSIDev,  
					count, offset, buffer, mode
				);
			break;

		}
		if (error != e_Err_NoError)
		{
			printf("Error occurred during DSP active communications.");
			return;
		}
	}
	// if reading from DSP
	else
	{
		switch(mode)
		{
		case kFlagDSPActiveComm_DSPDMA_HostInt:
		case kFlagDSPActiveComm_DSPIO_HostInt:
			error = 
				SI_PLXC33_ReadAddonInit
				(
					board->cgpSIDev,  
					count, offset, buffer, mode, callbackEvent, 
					addonInitBufferSize, addonInitBuffer
				);
			break;

		case kFlagDSPActiveComm_DSPDMA_HostPoll:
		case kFlagDSPActiveComm_DSPIO_HostPoll:
			error = 
				SI_PLXC33_ReadHostpoll
				(
					board->cgpSIDev,  
					count, offset, buffer, mode
				);
			break;
		
		}

		if (error != e_Err_NoError)
		{
			printf("Error occurred during DSP active communications.");
			return;
		}

		for (cnt=0; cnt<count; cnt++)
		{
			printf("0x%x\n", buffer[cnt]);//32-bit
		}	// close for
	}
*/

	return;
}


///////////////////////////////////////////////////////////////////////////////

void Sample_ConfigureFPGA()
{
//	PLXDevice PLXDevFPGA, *pPLXDevFPGA = &PLXDevFPGA;		

	char filename[256];
	INT32 error;

	printf("Enter the full path for the FPGA Configuration file:\r\n");
	scanf("%s", filename);
	getchar();

	error = 
		LoadBoardFPGAs(board.boardInfo.boardID, board.cgpSIDev, filename);
	if (error != 0)
	{
		return;
	}

	return;
}

/////////////////////////////////////////////////////////////////////////////

void PrintAvailableAddresss(int option)
{
	printf("\n");
/*
	switch (option)
	{
	case kGetAddress_HostView:
		printf("Available Add-on/Local Addresses:\n");
		printf("    SRAM0         = 0x000000 - 0x07FFFF\n");
		printf("    SRAM1         = 0x080000 - 0x0FFFFF\n");
		printf("    Daughter Card = 0x100000 - 0x17FFFF\n");
		printf("    CSR (Reset)   = 0x1F0000           \n");
		printf("    Int to DSP    = 0x1F0001           \n");
		break;

	case kGetAddress_DSPView:
		printf("Available DSP Addresses:\n");
		printf("    SRAM0 (CE0)   = 0x000000 - 0x3FFFFF\n");
		printf("    SRAM1 (CE1)   = 0x400000 - 0x7FFFFF\n");
//		printf("    DSP->Host Mem = 0xC00000 - 0xDFFFFF\n");
		printf("    Daughter Card = 0xFE0000 - 0xFEFFFF\n");
		printf("    PLX OpRegs    = 0xFF0000 - 0xFFFFFF\n");
		break;
	}
*/

	switch (option)
	{
	case kGetAddress_HostView:
		printf("Available Add-on/Local Addresses (byte boundary):\n");
		printf("    SRAM0             = 0x000000 - 0x1FFFFF\n");
		printf("    SRAM1             = 0x200000 - 0x3FFFFF\n");
		printf("    Expansion Card    = 0x780000 - 0x7BFFFF\n");
		printf("    Reset (CSR0)      = 0x7C0000           \n");
		printf("    Int to DSP (CSR1) = 0x7C0004           \n");
		break;

	case kGetAddress_DSPView:
		printf("Available DSP Addresses (DWord boundary):\n");
		printf("    SRAM0 (CE0)       = 0x000000 - 0x3FFFFF\n");
		printf("    SRAM1 (CE1)       = 0x400000 - 0x7FFFFF\n");
//		printf("    DSP->Host Mem (Addon-Init) = 0xC00000 - 0xDFFFFF\n");
		printf("    Expansion Card    = 0xFE0000 - 0xFEFFFF\n");
		printf("    PLX OpRegs        = 0xFF0000 - 0xFFFFFF\n");
		break;
	}

}

///////////////////////////////////////////////////////////////////////////////

#endif	//SIDEF_PLX_C3xC671x
///////////////////////////////////////////////////////////////////////////////

